/*
 * Copyright 2019-2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.mattbertolini.spring.web.servlet.mvc.bind.config;

import com.mattbertolini.spring.web.bind.introspect.ClassPathScanningAnnotatedRequestBeanIntrospector;
import com.mattbertolini.spring.web.bind.introspect.DefaultAnnotatedRequestBeanIntrospector;
import com.mattbertolini.spring.web.servlet.mvc.bind.BeanParameterMethodArgumentResolver;
import com.mattbertolini.spring.web.servlet.mvc.bind.DefaultPropertyResolverRegistry;
import com.mattbertolini.spring.web.servlet.mvc.bind.PropertyResolverRegistry;
import com.mattbertolini.spring.web.servlet.mvc.bind.resolver.RequestPropertyResolver;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerAdapter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

public class BinderConfiguration implements InitializingBean, BeanPostProcessor {
    private final Set<String> packagesToScan;
    private final PropertyResolverRegistry propertyResolverRegistry;
    private BeanParameterMethodArgumentResolver resolver;

    public BinderConfiguration() {
        this(new DefaultPropertyResolverRegistry());
    }

    public BinderConfiguration(PropertyResolverRegistry propertyResolverRegistry) {
        packagesToScan = new LinkedHashSet<>();
        this.propertyResolverRegistry = propertyResolverRegistry;
    }

    public BinderConfiguration addPackageToScan(String packageToScan) {
        packagesToScan.add(packageToScan);
        return this;
    }

    public BinderConfiguration setPackagesToScan(Set<String> packagesToScan) {
        this.packagesToScan.addAll(packagesToScan);
        return this;
    }

    public BinderConfiguration addResolver(RequestPropertyResolver resolver) {
        propertyResolverRegistry.addResolver(resolver);
        return this;
    }

    public BinderConfiguration addResolvers(Set<RequestPropertyResolver> resolvers) {
        propertyResolverRegistry.addResolvers(resolvers);
        return this;
    }

    public BinderConfiguration addResolvers(PropertyResolverRegistry propertyResolverRegistry) {
        this.propertyResolverRegistry.addResolvers(propertyResolverRegistry);
        return this;
    }

    public Set<String> getPackagesToScan() {
        return Collections.unmodifiableSet(packagesToScan);
    }

    @Override
    public void afterPropertiesSet() {
        DefaultAnnotatedRequestBeanIntrospector defaultIntrospector = new DefaultAnnotatedRequestBeanIntrospector(propertyResolverRegistry);
        ClassPathScanningAnnotatedRequestBeanIntrospector introspector = new ClassPathScanningAnnotatedRequestBeanIntrospector(defaultIntrospector, packagesToScan);
        try {
            introspector.afterPropertiesSet();
        } catch (Exception e) {
            throw new BeanInitializationException("Unable to create introspector");
        }
        resolver = new BeanParameterMethodArgumentResolver(introspector);
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        if (!(bean instanceof RequestMappingHandlerAdapter)) {
            return bean;
        }

        if (resolver == null) {
            throw new IllegalStateException("BeanParameterMethodArgumentResolver is null. Perhaps the afterPropertiesSet method was not called?");
        }
        RequestMappingHandlerAdapter adapter = (RequestMappingHandlerAdapter) bean;
        List<HandlerMethodArgumentResolver> currentResolvers = adapter.getCustomArgumentResolvers();
        if (currentResolvers == null) {
            currentResolvers = Collections.emptyList();
        }
        List<HandlerMethodArgumentResolver> newResolvers = new ArrayList<>(currentResolvers.size() + 1);
        newResolvers.addAll(currentResolvers);
        newResolvers.add(resolver);
        adapter.setCustomArgumentResolvers(newResolvers);

        return adapter;
    }
}
