/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.parseq;

import com.linkedin.parseq.HashManager;
import com.linkedin.parseq.HttpResponse;
import com.linkedin.parseq.Task;
import com.linkedin.parseq.exec.Exec;
import com.linkedin.parseq.function.Success;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GraphvizEngine {
    private static final Logger LOG = LoggerFactory.getLogger(GraphvizEngine.class);
    private final String _dotLocation;
    private final Path _cacheLocation;
    private final long _timeoutMs;
    private final HashManager _hashManager;
    private final Exec _exec;
    private final ConcurrentHashMap<String, Task<Exec.Result>> _inFlightBuildTasks;

    public GraphvizEngine(String dotLocation, Path cacheLocation, int cacheSize, long timeoutMs, int numThreads, long reaperDelayMs, int processQueueSize) {
        this._dotLocation = dotLocation;
        this._cacheLocation = cacheLocation;
        this._timeoutMs = timeoutMs;
        this._hashManager = new HashManager(this::removeCached, cacheSize);
        this._exec = new Exec(numThreads, reaperDelayMs, processQueueSize);
        this._inFlightBuildTasks = new ConcurrentHashMap();
    }

    public void start() {
        this._exec.start();
    }

    public void stop() {
        this._exec.stop();
    }

    public Task<HttpResponse> build(String hash, InputStream body) throws IOException {
        if (hash == null) {
            String content = "Missing hash.";
            LOG.info(content);
            return Task.value(new HttpResponse(400, content));
        }
        if (this._hashManager.contains(hash)) {
            LOG.info("hash found in cache: " + hash);
            return Task.value(new HttpResponse(200, ""));
        }
        if (body == null) {
            String content = "Missing body.";
            LOG.info(content);
            return Task.value(new HttpResponse(400, content));
        }
        if (this._dotLocation == null) {
            String content = "Missing dot.";
            LOG.info(content);
            return Task.value(new HttpResponse(500, content));
        }
        Task<Exec.Result> buildTask = this.getBuildTask(hash, body);
        return buildTask.transform("result", result -> {
            Integer status = null;
            String content = null;
            if (result.isFailed()) {
                status = 500;
                content = result.getError().toString();
            } else {
                switch (((Exec.Result)result.get()).getStatus()) {
                    case 0: {
                        this._hashManager.add(hash);
                        status = 200;
                        content = "";
                        break;
                    }
                    case 137: {
                        status = 500;
                        content = "graphviz process was killed because it did not finish within " + this._timeoutMs + "ms";
                        break;
                    }
                    default: {
                        status = 500;
                        content = this.writeGenericFailureInfo((Exec.Result)result.get());
                    }
                }
            }
            this._inFlightBuildTasks.remove(hash, buildTask);
            return Success.of(new HttpResponse(status, content));
        });
    }

    private Task<Exec.Result> getBuildTask(String hash, InputStream body) {
        Task<Exec.Result> existing = this._inFlightBuildTasks.get(hash);
        if (existing != null) {
            LOG.info("using in flight shareable: " + hash);
            return existing.shareable();
        }
        Task<Exec.Result> newBuildTask = this.createNewBuildTask(hash, body);
        existing = this._inFlightBuildTasks.putIfAbsent(hash, newBuildTask);
        if (existing != null) {
            LOG.info("using in flight shareable: " + hash);
            return existing.shareable();
        }
        return newBuildTask;
    }

    private Task<Exec.Result> createNewBuildTask(String hash, InputStream body) {
        LOG.info("building: " + hash);
        Task<Void> createDotFile = Task.action("createDotFile", () -> Files.copy(body, this.pathToCacheFile(hash, "dot"), StandardCopyOption.REPLACE_EXISTING));
        Task<Exec.Result> graphviz = this._exec.command("graphviz", this._timeoutMs, TimeUnit.MILLISECONDS, this._dotLocation, "-Tsvg", "-Grankdir=LR", "-Gnewrank=true", "-Gbgcolor=transparent", this.pathToCacheFile(hash, "dot").toString(), "-o", this.pathToCacheFile(hash, "svg").toString());
        Task<Exec.Result> graphvizWithTimeout = graphviz.withTimeout(this._timeoutMs * 2L, TimeUnit.MILLISECONDS);
        return createDotFile.andThen(graphvizWithTimeout);
    }

    private Path pathToCacheFile(String hash, String ext) {
        return this._cacheLocation.resolve(hash + "." + ext);
    }

    private File cacheFile(String hash, String ext) {
        return this.pathToCacheFile(hash, ext).toFile();
    }

    private void removeCached(String hash) {
        this.cacheFile(hash, "svg").delete();
        this.cacheFile(hash, "dot").delete();
    }

    private String writeGenericFailureInfo(Exec.Result result) throws IOException {
        StringBuilder sb = new StringBuilder();
        sb.append("graphviz process returned: ").append(result.getStatus()).append("\n").append("stdout:\n");
        Files.lines(result.getStdout()).forEach(sb::append);
        sb.append("stderr:\n");
        Files.lines(result.getStderr()).forEach(sb::append);
        return sb.toString();
    }
}

