/*
 * Copyright © 2016-2024 Lightbend, Inc. All rights reserved.
 * No information contained herein may be reproduced or transmitted in any form
 * or by any means without the express written permission of Lightbend, Inc.
 */

package com.lightbend.tools.fortify.plugin

// so we can generate buildid.scasession.increment, with an entry
// for each source file we processed

import java.io.{File, PrintWriter}
import java.text.SimpleDateFormat
import java.util.TimeZone

object SessionWriter {

  case class Entry(sourcePath: File, nstPath: File, lines: Int)

  def formatDate(millis: Long, timeZone: TimeZone = TimeZone.getDefault): String = {
    val format = new SimpleDateFormat("yyy-MM-dd'T'HH:mm:ss.SSSXXX")
    format.setTimeZone(timeZone)
    format.format(millis)
  }

  def escape(s: String): String = s.flatMap {
    case '&' => "&amp;"
    case '<' => "&lt;"
    case c   => c.toString
  }

  def write(
      w: PrintWriter,
      buildId: String,
      session: Seq[Entry],
      encoding: String,
      millis: Long = System.currentTimeMillis(),
      timeZone: TimeZone = TimeZone.getDefault): Unit = {
    w.println(s"""<>""")
    w.println(s"""<?xml version="1.0" encoding="UTF-8"?>""")
    val now = formatDate(millis, timeZone)
    w.println(s"""<BuildSession xmlns="xmlns://www.fortifysoftware.com/schema/build-session">""")
    w.println(s"""  <FormatVersion schema="1.3" engine="scala-fortify"/>""")
    w.println(s"""  <TimeStamp created="$now" lastModified="$now"/>""")
    w.println(s"""  <Id>${escape(buildId)}</Id>""")
    w.println(s"""  <FilePool>""")
    for ((Entry(sourcePath, _, lines), index) <- session.zipWithIndex) {
      w.println(s"""    <FilePoolEntry id="$index">""")
      w.println(s"""      <File type="scala" lines="$lines" encoding="$encoding">""")
      w.println(s"""        <Size>${sourcePath.length}</Size>""")
      w.println(s"""        <LastModified>${sourcePath.lastModified}</LastModified>""")
      w.println(s"""        <Path>${escape(sourcePath.getCanonicalPath)}</Path>""")
      w.println(s"""      </File>""")
      w.println(s"""    </FilePoolEntry>""")
    }
    for ((Entry(_, nstPath, lines), index) <- session.zipWithIndex) {
      w.println(s"""    <FilePoolEntry id="${(session.size + index)}">""")
      w.println(s"""      <File type="nst" lines="$lines">""")
      w.println(s"""        <Path>${escape(nstPath.getCanonicalPath)}</Path>""")
      w.println(s"""        <Source id="$index"/>""")
      w.println(s"""        <SecondarySources/>""")
      w.println(s"""      </File>""")
      w.println(s"""    </FilePoolEntry>""")
    }
    w.println(s"""  </FilePool>""")
    w.println(s"""  <Entries>""")
    w.println(s"""    <Entry>""")
    w.println(s"""      <Sources>""")
    for (n <- 0 until (session.size * 2))
      w.println(s"""        <FileRef id="$n"/>""")
    w.println(s"""      </Sources>""")
    w.println(s"""    </Entry>""")
    w.println(s"""  </Entries>""")
    w.println(s"""  <ClassPath/>""")
    w.println(s"""  <JavaBuildDirectories/>""")
    w.println(s"""  <LibDirs/>""")
    w.println(s"""</BuildSession>""")
  }

}
