/*
 * Copyright © 2016-2024 Lightbend, Inc. All rights reserved.
 * No information contained herein may be reproduced or transmitted in any form
 * or by any means without the express written permission of Lightbend, Inc.
 */

package com.lightbend.tools.fortify.plugin

import java.io.File
import java.nio.file.PathMatcher

trait PluginOptions {

  /** The output directory to write to, or None for cwd */
  var outputDir: Option[File] = None

  /** The build id. If set, also determines the output directory, unless overridden later with
    * "out=". If not set, NST is generated, but no session files.
    */
  var buildId: Option[String] = None

  var licensePath =
    java.nio.file.Paths.get(
      System.getProperty("user.home"),
      ".lightbend",
      "license"
    ).toFile
  var scaVersion = "24.2"

  /** Path specifiers for source files to exclude */
  var excludes: Vector[PathMatcher] = Vector()

  // remaining options are undocumented
  var showSourceInfo = true
  var suppressEntitlementCheck = false
  var trace = false
}

object PluginOptions {
  implicit class RegexInterpolator(sc: StringContext) {
    def r = VersionSpecificHelpers.makeRegex(
      sc.parts.mkString, sc.parts.tail.map(_ => "x"))
  }
  val HelpString =
    """|  -P:fortify:build=<name>         Build id for output files (default: none)"
       |  -P:fortify:scaversion=<number>  SCA version (default: 24.2), used when
       |                                  build id support is enabled to write files to
       |                                  the right directory under ~/.fortify,
       |                                  for example: ~/.fortify/sca24.2
       |  -P:fortify:out=<path>           Directory for output files (default:
       |                                  determined by build id; otherwise, cwd)
       |  -P:fortify:license=<path>       Full path to Lightbend license file
       |                                  (default: ~/.lightbend/license)
       |  -P:fortify:exclude=<path_specs> Excludes some source files from translation.
       |                                  Path specifiers may use glob syntax.
       |                                  Separate multiple path specifiers with
       |                                  colons (Linux, MacOS) or semicolons (Windows).
       |""".stripMargin
  def pathForBuildId(scaVersion: String, buildId: String): java.io.File = {
    val home = System.getProperty("user.home")
    val base =
      if (scala.util.Properties.isWin)
        s"AppData/Local/Fortify/sca$scaVersion/build"
      else
        s".fortify/sca$scaVersion/build"
    new File(s"$home/$base/$buildId")
  }
}
