/*
 * Copyright © 2016-2024 Lightbend, Inc. All rights reserved.
 * No information contained herein may be reproduced or transmitted in any form
 * or by any means without the express written permission of Lightbend, Inc.
 */

package com.lightbend.tools.fortify.plugin

import java.io.File
import java.nio.charset.Charset
import java.nio.file.Files
import java.time.ZonedDateTime

import scala.util.control.NonFatal

import com.lightbend.prodsuite.licenses.LicenseParser

object LicenseChecker {

  def verifyEntitlements(
      licensePath: File,
      suppressEntitlementCheck: Boolean,
      error: String => Unit): Unit = {
    if (suppressEntitlementCheck)
      return
    BuildInfo.checkGrant.foreach { grant =>
      try {
        val (user, exp) = validateGrant(licensePath, grant)
        val when = exp.map(_.toString).getOrElse("never")
        println(
          s"scala-fortify ${BuildInfo.version}, licensed to $user (expires: $when)")
      }
      catch {
        case NonFatal(ex) =>
          error("License validation failed:")
          throw ex
      }
    }
  }

  private def validateGrant(path: File, grant: String): (String, Option[ZonedDateTime]) = {
    val esl =
      try {
        val bytes = Files.readAllBytes(path.toPath)
        LicenseParser.parse(new String(bytes, Charset.forName("UTF-8")))
      }
      catch {
        case NonFatal(ex) =>
          throw new RuntimeException(s"Error validating license file `$path`", ex)
      }
    if (!esl.getGrants.contains(grant))
      throw new RuntimeException(
        s"License file `$path` does not contain the required grant '$grant' ($esl)")
    val exp = Option(esl.getExpiry.orElse(null))
    exp.foreach { expires =>
      if (expires.isBefore(ZonedDateTime.now()))
        throw new RuntimeException(
          s"License file `$path` expired at $expires ($esl)")
    }
    (esl.getUser, exp)
  }

}
