/*
 * Copyright © 2016-2024 Lightbend, Inc. All rights reserved.
 * No information contained herein may be reproduced or transmitted in any form
 * or by any means without the express written permission of Lightbend, Inc.
 */

package com.lightbend.tools.fortify.plugin

import java.io.File

import scala.tools.nsc

import nsc.Global
import nsc.plugins.{Plugin, PluginComponent}

class FortifyPlugin(val global: Global) extends Plugin {

  override val name = "fortify"
  override val description = "compile Scala to Fortify NST"

  object component extends FortifyComponent {
    override val global = FortifyPlugin.this.global
    override val runsAfter = List("mixin")
    override val runsBefore = List("cleanup")
  }

  override val components: List[PluginComponent] =
    if (global.settings.isScaladoc) Nil
    else List(component)

  override val optionsHelp: Option[String] = Some(PluginOptions.HelpString)

  override def init(options: List[String], error: String => Unit) = {
    import PluginOptions.RegexInterpolator
    options.foreach {
      case r"license=(.*)$path" =>
        component.licensePath = new File(path)
      case r"build=(.*)$id" =>
        component.buildId = Some(id)
      case r"scaversion=(.*)$version" =>
        component.scaVersion = version
      case r"out=(.*)$out" =>
        component.outputDir = Some(new File(out))
      case r"exclude=(.*)$specs" =>
        component.excludes ++= Paths.parseExcludes(specs)
      // remaining options are undocumented
      case r"showSourceInfo=(.*)$show" =>
        component.showSourceInfo = show.toBoolean
      case r"suppressEntitlementCheck=(.*)$suppress" =>
        component.suppressEntitlementCheck = suppress.toBoolean
      case r"trace=(.*)$trace" =>
        component.trace = trace.toBoolean
      case arg =>
        error(s"Bad argument: $arg")
    }
    for (id <- component.buildId)
      if (!component.outputDir.isDefined)
        component.outputDir =
          Some(PluginOptions.pathForBuildId(component.scaVersion, id))
    LicenseChecker.verifyEntitlements(
      component.licensePath,
      component.suppressEntitlementCheck,
      error)
    for (id <- component.buildId; dir <- component.outputDir) {
      dir.mkdirs()
      new java.io.File(dir.getParent, s"$id.scasession.lock")
        .createNewFile()
    }
    true
  }
}
