/*
 * Copyright © Lightbend, Inc. dba Akka
 * All rights reserved.
 * No information contained herein may be reproduced or transmitted in any form
 * or by any means without the express written permission of Akka, Inc.
 */

package com.lightbend.tools.fortify.plugin

class Twirl(content: String) {

  /** If this Scala source contains a Twirl source map, return a Some of the template file path,
    * else return None.
    */
  def originalPath: Option[String] =
    meta.get("SOURCE")

  // the remainder of the code is copy-and-pasted from TwirlCompiler.scala
  // in playframework/twirl.  it's simple code that rarely changes, so
  // copy-and-paste seems okay to me, since the alternative is to put Twirl
  // on the user's compile-time classpath, yuck to that.

  lazy val meta: Map[String, String] = {
    val Meta = """([A-Z]+): (.*)""".r
    val UndefinedMeta = """([A-Z]+):""".r
    Map.empty[String, String] ++ {
      try {
        content.split("-- GENERATED --")(1).trim.split('\n').map { m =>
          m.trim match {
            case Meta(key, value)   => (key -> value)
            case UndefinedMeta(key) => (key -> "")
            case _                  => ("UNDEFINED", "")
          }
        }.toMap
      }
      catch {
        case _: Exception => Map.empty[String, String]
      }
    }
  }

  lazy val matrix: Seq[(Int, Int)] = {
    for (pos <- meta("MATRIX").split('|').toSeq; c = pos.split("->"))
      yield try {
        Integer.parseInt(c(0)) -> Integer.parseInt(c(1))
      }
      catch {
        case _: Exception => (0, 0) // Skip if MATRIX meta is corrupted
      }
  }

  lazy val lines: Seq[(Int, Int)] = {
    for (pos <- meta("LINES").split('|').toSeq; c = pos.split("->"))
      yield try {
        Integer.parseInt(c(0)) -> Integer.parseInt(c(1))
      }
      catch {
        case _: Exception => (0, 0) // Skip if LINES meta is corrupted
      }
  }

  def mapPosition(generatedPosition: Int): Int = {
    matrix.indexWhere(p => p._1 > generatedPosition) match {
      case 0 => 0
      case i if i > 0 => {
        val pos = matrix(i - 1)
        pos._2 + (generatedPosition - pos._1)
      }
      case _ => {
        val pos = matrix.takeRight(1)(0)
        pos._2 + (generatedPosition - pos._1)
      }
    }
  }

  def mapLine(generatedLine: Int): Int = {
    lines.indexWhere(p => p._1 > generatedLine) match {
      case 0 => 0
      case i if i > 0 => {
        val line = lines(i - 1)
        line._2 + (generatedLine - line._1)
      }
      case _ => {
        val line = lines.takeRight(1)(0)
        line._2 + (generatedLine - line._1)
      }
    }
  }

}
