/**
 * 
 */
package com.ibm.storage.clientlibrary;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;



/**
 * @author ArunIyengar 
 * 
 */
/*
 * Utility methods called by several other classes
 */
public class Util {

    /**
     * Perform an unchecked cast while suppressing warnings
     * 
     * @param obj
     *            object to be cached
     * @param <T>
     *            type of cast object
     * @return cast object
     * 
     * */
    @SuppressWarnings({"unchecked"})
    public static <T> T uncheckedCast(Object obj) {
        return (T) obj;
    }
    
    /**
     * compress a serializable object using gzip
     * 
     * @param object
     *            object which implements Serializable
     * @return byte array containing compressed objects
     * 
     * */
    public static byte[] compress(Serializable object) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            GZIPOutputStream gzipOut = null;
            ObjectOutputStream objectOut = null;
            try {
                gzipOut = new GZIPOutputStream(baos);
                objectOut = new ObjectOutputStream(gzipOut);
                objectOut.writeObject(object);
            }
            catch (Exception e) {
                describeException(e, "Exception in Util.compress");
            }
            finally {
                objectOut.close();
                gzipOut.close();
            }
        }
        catch (Exception e) {
            describeException(e, "Exception in Util.compress");
        }
        return baos.toByteArray();
    }

    /**
     * Decompress a compressed object
     * 
     * @param bytes
     *            byte array corresponding to compressed object
     * @param <T>
     *            type of decompressed object
     * @return decompressed object
     * 
     * */
    public static <T> T decompress(byte[] bytes) {
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        T object = null;
        try {
            GZIPInputStream gzipIn = null;
            ObjectInputStream objectIn = null;
            try  {
                gzipIn = new GZIPInputStream(bais);
                objectIn = new ObjectInputStream(gzipIn);
                object = uncheckedCast(objectIn.readObject());
            }
            catch (Exception e) {
                describeException(e, "Exception in Util.decompress");
            }
            finally {
                objectIn.close();
                gzipIn.close();
            }       
        }
        catch (Exception e) {
            describeException(e, "Exception in Util.decompress");
        }
        return object;
    }
    
    
    /**
     * Output information about an exception
     * 
     * @param e
     *            The exception
     * @param message
     *            Message to output
     * 
     * */
    public static void describeException(Exception e, String message) {
        System.out.println(message);
        System.out.println(e.getMessage());
        e.printStackTrace();
    }
    
    /**
     * Return the current time
     * 
     * @return Milliseconds since January 1, 1970
     * 
     * */
      public static long getTime() {
          return System.currentTimeMillis();
        // return (new Date()).getTime();
        //java.util.Date.getTime() method returns how many milliseconds have passed since 
        //January 1, 1970, 00:00:00 GMT
    }
}
