/*
* © Copyright IBM Corp. 2016
* All Rights Reserved. US Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
*/

package com.ibm.mfp.server.security.external.resource;

import com.ibm.mfp.server.registration.external.model.AuthenticatedUser;
import com.ibm.mfp.server.registration.external.model.ClientData;

import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.Map;

/**
 * Security context of an adapter REST call. Provides access to the security context of the client, and the client registration data.
 * Using this interface an adapter can:
 * <br/><ul>
 * <li> Get the information of the authenticated user </li>
 * <li> Get introspection data of the access token </li>
 * <li> Get the registration data of the current client, or any registered client, including the details of the mobile device,
 * users associated with the application instance and custom client attributes </li>
 * <li> Modify client registration data </li>
 * <li> Find registered clients using various search criteria </li>
 * <li> Validate credentials with a security check </li>
 * </ul>
 *
 * <br/>
 * To get access to the security context in an adapter, use the <code>@Context</code> annotation to inject it.
 * For example:
 *
 * <pre>
 * <code>
 * &#64;Context
 * AdapterSecurityContext securityContext;
 *
 * &#64;OAuthSecurity(scope = "userLogin")
 * &#64;GET
 * &#64;Produces(MediaType.TEXT_PLAIN)
 * public String sayHello() {
 *    AuthenticatedUser user = securityContext.getAuthenticatedUser();
 *    return "Hello " + user.getDisplayName();
 * }
 * </code>
 * </pre>
 *
 * @author artem
 */
public interface AdapterSecurityContext {

    /**
     * Gets the latest authenticated user.
     *
     * @return authenticated user, null if there's no authenticated user at the moment
     */
    AuthenticatedUser getAuthenticatedUser();

    /**
     * Gets introspection data of the current access token provided in the request. <br/>
     * If the resource is unprotected and there's no access token, this method returns null. <br/>
     * If the token has expired or has been revoked, this method throws a runtime exception which is automatically translated into
     * an error response according to OAuth 2.0.
     *
     * @return introspection data of the current access token, or null if this is an unprotected resource
     */
    TokenIntrospectionData getAccessTokenInfo();

    /**
     * Gets registration data for the caller mobile client. <br/>
     * If the caller is not a mobile client, or the resource is unprotected and the caller is unknown, this method returns null.
     *
     * @return the registration data of the caller mobile client, or null if the caller is not a mobile client or the caller is unknown
     */
    ClientData getClientRegistrationData();

    /**
     * Gets registration data for the given client ID. Returns null if the client does not exist.
     *
     * @param clientId the client ID
     * @return the client registration data, or null if the client does not exist.
     */
    ClientData getClientRegistrationData(String clientId);

    /**
     * Finds the registration data of mobile clients by the given search criteria.
     *
     * @param criteria the search criteria
     * @return list of clients satisfying the given criteria
     */
    List<ClientData> findClientRegistrationData(ClientSearchCriteria criteria);

    /**
     * Updates the registration data of a client. <br/>
     * The <code>ClientData</code> isnstance must be obtained via {@link #getClientRegistrationData()}, {@link #getClientRegistrationData(String)},
     * or {@link #findClientRegistrationData(ClientSearchCriteria)}. <br/>
     * If the registration data has been modified by a concurrent request after it was obtained by this request,
     * this method throws a runtime exception which is automatically reported to the client as HTTP status 409 (Conflict).
     *
     * @param clientData the modified data to be stored.
     */
    void storeClientRegistrationData(ClientData clientData);

    /**
     * Validates the given credentials with the given security check. <br/>
     * The security check may also use information provided in the HTTP request (e.g., request headers) to authorize the request. <br/>
     * The security check is created and initialized with its default configuration.
     * The state of the security check is not preserved after this method returns. <br/>
     * If the security check cannot be found or created, this method throws a RuntimeException.
     *
     * @param securityCheckName check name
     * @param credentials       credentials (name-value pairs) to be validated by the security check
     * @param request           the client request
     * @return true if the credentials have been validated successfully, false otherwise
     * @throws RuntimeException if the check does not exist or cannot be instantiated
     */
    boolean validateCredentials(String securityCheckName, Map<String, Object> credentials, HttpServletRequest request);
}
