/*
* © Copyright IBM Corp. 2016
* All Rights Reserved. US Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
*/

package com.ibm.mfp.server.security.external.checks;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Field annotation that defines a reference to a security check, creating a dependency between the checks.<br/>
 * The annotation is applied to a field definition within a security-check class. The type of the annotated field should
 * be the class of the referenced security check, and the field must be defined as transient.<br/>
 * If there is more than one security check of the same class, use the annotation's {@link #name} element to specify
 * the name of the target security check.<br/>
 * The referenced security check must be available in the same adapter as the referencing class.<br/>
 * If the security framework cannot identify a matching security check, or if the dependency validation fails, the
 * adapter fails to deploy.
 * <br/>
 * <br/>
 * <b>Example</b><br/>
 * The following code defines a {@code UserLoginSecurityCheckReference} field that references a
 * {@code UserLogin} security check of the {@code userLoginSecurityCheck} class:<br/>
 * {@code @SecurityCheckReference(name = "UserLogin")}<br/>
 * {@code private transient UserLoginSecurityCheck userLoginSecurityCheckReference;}
 *
 * @author artem
 *         Date: 8 Aug, 2015
 */
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface SecurityCheckReference {

    /**
     * The name of the referenced security check. The default value is an empty string, which indicates that there is
     * a single security check of the referenced class.<br/>
     * If there is more than one security-check definition of the referenced class, set {@code name}
     * to the name of your target security check. Otherwise, the adapter deployment fails.
     *
     * @return The name of the referenced security check or an empty string.
     */
    String name() default "";
}

