/*
* © Copyright IBM Corp. 2016
* All Rights Reserved. US Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
*/

package com.ibm.mfp.server.security.external.checks;

import javax.servlet.http.HttpServletRequest;
import java.io.Externalizable;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

/**
 * Represents server-side state of a security check.<br/>
 * Security checks are kept in a distributed cache, and are stateful, so it's the responsibility of the implementor
 * to handle the persistent state.
 *
 * @author artem
 *         Date: 6/28/15
 */

public interface SecurityCheck extends Externalizable {

    /**
     * Create configuration object and populate fields from the given properties.
     * Use errors and warnings maps to report problems.<br/>
     * Called during deployment.
     *
     * @param properties the merged properties to read and validate
     * @return new configuration object, not null
     */
    SecurityCheckConfiguration createConfiguration(Properties properties);

    /**
     * Initialize the security check with the context and configuration properties.
     * This method is called after instantiation and also on each retrieve.
     * The security checks should not keep the configuration data in the persistent state
     *
     * @param name                 name of the security check
     * @param config               security check configuration created by {@link #createConfiguration(Properties)}
     * @param authorizationContext provides access to the transient state of the calling client
     * @param registrationContext  provides access to the persistent state of the calling client
     */
    void setContext(String name, SecurityCheckConfiguration config, AuthorizationContext authorizationContext, RegistrationContext registrationContext);

    /**
     * Get the security check inactivity timeout, usually the configured value.
     * 0 means no inactivity timeout defined for this check
     *
     * @return inactivity timeout in seconds, 0 if none
     */
    int getInactivityTimeoutSec();

    /**
     * Get the expiration of the current state of the security check.
     * No assumption is made about the meaning of the state (success, failure, or any other).
     * After the expiration the state of the security check is lost.
     *
     * @return the time of expiration in millis
     */
    long getExpiresAt();

    /**
     * Request the given scope from this security check.
     * The check can return success, challenge, or failure
     *
     * @param scope       the requested scope, comes from the scope mapping processing.
     *                    Usually should not be analyzed by the security check, except of the case of a custom check
     *                    that knows exactly what scopes should be granted at any point.
     * @param credentials the credentials sent by the client - either as a challenge response or pre-emptively
     * @param request     the pre-authorization or registration request sent by the client
     * @param response    the response to which this check adds its success, challenge, or failure
     */
    void authorize(Set<String> scope, Map<String, Object> credentials, HttpServletRequest request, AuthorizationResponse response);

    /**
     * Make sure this check currently grants the requested scope.<br/>
     * If the scope is granted, the implementation should add the granted scope, its expiration, and custom introspection data to the response parameter.
     * If the scope is not granted, the implementation should return silently.
     *
     * @param scope    scope expected to be granted by this check
     * @param response the response to which this check adds its granted scope and custom introspection data
     */
    void introspect(Set<String> scope, IntrospectionResponse response);

    /**
     * Called upon explicit logout from this security check.
     * The implementation may choose to modify or delete its persistent (registered) state, or other custom logic.<br/>
     * The transient state of this security check is destroyed automatically.
     */
    void logout();
}
