/*
* © Copyright IBM Corp. 2016
* All Rights Reserved. US Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
*/

package com.ibm.mfp.server.security.external.checks;

import java.util.Map;
import java.util.Set;

/**
 * Aggregates authorization responses from multiple security checks.
 * Challenge eliminates the success (scopes), and failure eliminates success and challenge
 *
 * @author artem on 11/19/15.
 */
public interface AuthorizationResponse {

    enum ResponseType {SUCCESS, FAILURE, CHALLENGE}

    /**
     * Get the current type of the response
     *
     * @return SUCCESS, CHALLENGE, or FAILURE
     */
    ResponseType getType();

    /**
     * Add success scope and its expiration
     *
     * @param scope             the scope granted by the security check
     * @param expiresAt         expiration of the granted scope
     * @param securityCheckName name of the security check for the success data object
     * @return the same response object
     */
    AuthorizationResponse addSuccess(Set<String> scope, long expiresAt, String securityCheckName);

    /**
     * Add success scope, its expiration, and a field for the success object that will be delivered to the challenge handler.
     * This method can be called repeatedly, and the fields will be added to the success data object.
     *
     * @param scope             the scope granted by the security check
     * @param expiresAt         expiration of the granted scope
     * @param securityCheckName name of the security check for the success data object
     * @param key               the field name in the success data object
     * @param value             the field value
     * @return the same response object
     */
    AuthorizationResponse addSuccess(Set<String> scope, long expiresAt, String securityCheckName, String key, Object value);

    /**
     * Add a challenge.
     *
     * @param securityCheckName the name of the challenging security check
     * @param challenge         the challenge object
     * @return the same response object
     */
    AuthorizationResponse addChallenge(String securityCheckName, Map<String, Object> challenge);

    /**
     * Add a failure
     *
     * @param securityCheckName the name of the failed security check
     * @param failureData       the failure object
     * @return the same response object
     */
    AuthorizationResponse addFailure(String securityCheckName, Map<String, Object> failureData);
}
