/*
* © Copyright IBM Corp. 2016
* All Rights Reserved. US Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
*/

package com.ibm.mfp.server.registration.external.model;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;

import java.security.interfaces.RSAPublicKey;
import java.util.Map;

/**
 * Data of a registered client.  <br/>
 * The class constructors are reserved for internal use only. Adapters can use the methods getClientRegistrationData or
 * findClientRegistrationData in AdapterSecurityContext to get instances of this class. Adapters can change the
 * registration data stored in this class (for examplem add public or protected attributes). After making changes,
 * adapters must call AdapterSecurityContext.storeClientRegistrationData() to store the changes.
 *
 * @author artem
 *         Date: 6/30/15
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class ClientData {

    private String clientId;
    private RegistrationData registration;
    private boolean enabled;
    private String signatureAlgorithm;
    private Map<String, Object> webKey;
    @JsonIgnore
    private RSAPublicKey publicKey;
    private boolean registrationComplete;
    private long lastActivityTime;
    private Map<String, AuthenticatedUser> users;

    private PersistentAttributes publicAttributes;
    private PersistentAttributes protectedAttributes;

    /**
     * Reserved for internal use only.
     */
    public ClientData() {
    }

    /**
     * Reserved for internal use only.
     */
    public ClientData(String clientId, RegistrationData registration,
                      boolean enabled,
                      boolean registrationComplete,
                      long lastActivityTime, String signatureAlgorithm, Map<String, Object> webKey,
                      Map<String, AuthenticatedUser> users,
                      PersistentAttributes publicAttributes,
                      PersistentAttributes protectedAttributes,
                      RSAPublicKey publicKey) {
        this.clientId = clientId;
        this.registration = registration;
        this.enabled = enabled;
        this.registrationComplete = registrationComplete;
        this.lastActivityTime = lastActivityTime;
        this.signatureAlgorithm = signatureAlgorithm;
        this.webKey = webKey;
        this.publicKey = publicKey;
        this.users = users;
        this.publicAttributes = publicAttributes;
        this.protectedAttributes = protectedAttributes;
    }

    /**
     * Gets the unique ID of the application instance.
     *
     * @return client ID
     */
    public String getClientId() {
        return clientId;
    }

    /**
     * Gets the data reported by the mobile client during the registration, including the application and the device it is installed on.
     *
     * @return mobile client registration data
     */
    public RegistrationData getRegistration() {
        return registration;
    }

    /**
     * Returns true if the application instance is enabled, false otherwise.
     *
     * @return true if the application instance is enabled, false otherwise
     */
    public boolean isEnabled() {
        return enabled;
    }

    /**
     * Gets the name of the signature algorithm used by the application instance.
     *
     * @return algorithm name
     */
    public String getSignatureAlgorithm() {
        return signatureAlgorithm;
    }

    /**
     * Returns a JSON representation of the public key used by the application instance.
     * Returns null if the client has no public key
     *
     * @return JSON representation of the public key; null if the client has no public key
     * @see #getPublicKey()
     */
    public Map<String, Object> getWebKey() {
        return webKey;
    }

    /**
     * Gets the public key used by the application instance.
     * Null if the client has no public key.
     *
     * @return the public key, may be null
     * @see #getWebKey()
     */
    public RSAPublicKey getPublicKey() {
        return publicKey;
    }

    /**
     * Returns true if the registration is complete, false if the registration is in progress
     *
     * @return true if the registration is complete, false if the registration is in progress
     */
    public boolean getRegistrationComplete() {
        return registrationComplete;
    }

    /**
     * Gets the timestamp of the last registered client activity, precise up to an hour. The timestamp is provided
     * as Unix time, that is, milliseconds since Unix Epoch.
     *
     * @return time of last recorded client activity (in milliseconds)
     */
    public long getLastActivityTime() {
        return lastActivityTime;
    }

    /**
     * Gets the users associated with the application instance.
     *
     * @return map with the security check name as a key, and authenticated user as value
     */
    public Map<String, AuthenticatedUser> getUsers() {
        return users;
    }

    /**
     * Gets the custom public attributes.
     *
     * @return modifyable the public attributes
     */
    public PersistentAttributes getPublicAttributes() {
        return publicAttributes;
    }

    /**
     * Gets the custom protected attributes.
     *
     * @return modifyable the protected attributes
     */
    public PersistentAttributes getProtectedAttributes() {
        return protectedAttributes;
    }
}
