/*
* © Copyright IBM Corp. 2016
* All Rights Reserved. US Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
*/

package com.ibm.mfp.adapter.api;

import com.ibm.json.java.JSONObject;
import com.worklight.adapters.rest.api.MFPServerOAuthException;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;

import javax.ws.rs.core.Application;
import java.io.IOException;

/**
 * API to retrieve information about the current adapter and send REST requests to other adapters.
 * This API is accessible using the Jax-RS {@literal@}Context annotation, for example by defining a member varaible:
 * <pre>
 <code>
 {@literal@}Context
 private AdaptersAPI adaptersApi;
 }
 </code>
 </pre>
 *
 */
public interface AdaptersAPI {

	/**
	 * Sends HTTP request to another REST or Javascript adapter.
	 * 
	 * @param request - Apache {@link HttpUriRequest} to be sent to the adapter. The URI of this request must not be
	 * absolute. It must be relative to the "/adapters" endpoint. For example: "/myadapter/getStories" 
	 * @return
	 * The response (standard {@link HttpResponse})
	 * @throws IOException
	 * @throws MFPServerOAuthException
	 */
	 HttpResponse executeAdapterRequest(HttpUriRequest request) throws IOException;
	
	/**
	 * Reads the response to JSON. It assumes that the response contains a JSON content.
	 * 
	 * @param response - The response to be read
	 * @return A JSON object with the response content
	 * @throws IOException
	 */
	 JSONObject getResponseAsJSON(HttpResponse response) throws IOException;
	
	/**
	 * Helper method to create request to a JS adapter (via REST)
	 * 
	 * @param adapter - The adapter name
	 * @param procedure - The procedure name
	 * @param args - Arguments to be passed (Optional)
	 * @return the request for javascript adapter invocation
	 */
	 HttpUriRequest createJavascriptAdapterRequest(String adapter, String procedure, Object... args);

	/**
	 * Returns the name of the current adapter
	 * @return name of the adapter
     */
	 String getAdapterName();

	/**
	 * Returns the Jax-RS application class of the current adapter, casted to the given class. The applicationClass argument must not be null, and must also
	 * be assignable from the adapter's application class.
	 * @return application instance
	 * @throws IllegalArgumentException if applicationClass is null or not applicable
     */
	<T> T getJaxRsApplication(Class<T> applicationClass);
}
