/*
 * (C) Copyright IBM Corp. 2020.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.ibm.cloud.ibm_key_protect_api.v2.model;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;

import com.ibm.cloud.sdk.core.service.model.GenericModel;

/**
 * The restoreKey options.
 */
public class RestoreKeyOptions extends GenericModel {

  /**
   * Alters server behavior for POST or DELETE operations. A header with `return=minimal` causes the service to  return
   * only the key identifier, or metadata. A header containing `return=representation` returns both the key  material
   * and metadata in the response entity-body. If the key has been designated as a root key, the  system cannot return
   * the key material.
   *
   * **Note:** During POST operations, Key Protect may not immediately return the key material due to key generation
   * time. To retrieve the key material, you can perform a subsequent `GET /keys/{id}` request.
   */
  public interface Prefer {
    /** return=representation. */
    String RETURN_REPRESENTATION = "return=representation";
    /** return=minimal. */
    String RETURN_MINIMAL = "return=minimal";
  }

  protected String id;
  protected String bluemixInstance;
  protected InputStream keyRestoreBody;
  protected String correlationId;
  protected String prefer;

  /**
   * Builder.
   */
  public static class Builder {
    private String id;
    private String bluemixInstance;
    private InputStream keyRestoreBody;
    private String correlationId;
    private String prefer;

    private Builder(RestoreKeyOptions restoreKeyOptions) {
      this.id = restoreKeyOptions.id;
      this.bluemixInstance = restoreKeyOptions.bluemixInstance;
      this.keyRestoreBody = restoreKeyOptions.keyRestoreBody;
      this.correlationId = restoreKeyOptions.correlationId;
      this.prefer = restoreKeyOptions.prefer;
    }

    /**
     * Instantiates a new builder.
     */
    public Builder() {
    }

    /**
     * Instantiates a new builder with required properties.
     *
     * @param id the id
     * @param bluemixInstance the bluemixInstance
     */
    public Builder(String id, String bluemixInstance) {
      this.id = id;
      this.bluemixInstance = bluemixInstance;
    }

    /**
     * Builds a RestoreKeyOptions.
     *
     * @return the new RestoreKeyOptions instance
     */
    public RestoreKeyOptions build() {
      return new RestoreKeyOptions(this);
    }

    /**
     * Set the id.
     *
     * @param id the id
     * @return the RestoreKeyOptions builder
     */
    public Builder id(String id) {
      this.id = id;
      return this;
    }

    /**
     * Set the bluemixInstance.
     *
     * @param bluemixInstance the bluemixInstance
     * @return the RestoreKeyOptions builder
     */
    public Builder bluemixInstance(String bluemixInstance) {
      this.bluemixInstance = bluemixInstance;
      return this;
    }

    /**
     * Set the keyRestoreBody.
     *
     * @param keyRestoreBody the keyRestoreBody
     * @return the RestoreKeyOptions builder
     */
    public Builder keyRestoreBody(InputStream keyRestoreBody) {
      this.keyRestoreBody = keyRestoreBody;
      return this;
    }

    /**
     * Set the correlationId.
     *
     * @param correlationId the correlationId
     * @return the RestoreKeyOptions builder
     */
    public Builder correlationId(String correlationId) {
      this.correlationId = correlationId;
      return this;
    }

    /**
     * Set the prefer.
     *
     * @param prefer the prefer
     * @return the RestoreKeyOptions builder
     */
    public Builder prefer(String prefer) {
      this.prefer = prefer;
      return this;
    }

    /**
     * Set the keyRestoreBody.
     *
     * @param keyRestoreBody the keyRestoreBody
     * @return the RestoreKeyOptions builder
     *
     * @throws FileNotFoundException if the file could not be found
     */
    public Builder keyRestoreBody(File keyRestoreBody) throws FileNotFoundException {
      this.keyRestoreBody = new FileInputStream(keyRestoreBody);
      return this;
    }
  }

  protected RestoreKeyOptions(Builder builder) {
    com.ibm.cloud.sdk.core.util.Validator.notEmpty(builder.id,
            "id cannot be empty");
    com.ibm.cloud.sdk.core.util.Validator.notNull(builder.bluemixInstance,
            "bluemixInstance cannot be null");
    id = builder.id;
    bluemixInstance = builder.bluemixInstance;
    keyRestoreBody = builder.keyRestoreBody;
    correlationId = builder.correlationId;
    prefer = builder.prefer;
  }

  /**
   * New builder.
   *
   * @return a RestoreKeyOptions builder
   */
  public Builder newBuilder() {
    return new Builder(this);
  }

  /**
   * Gets the id.
   *
   * The v4 UUID that uniquely identifies the key.
   *
   * @return the id
   */
  public String id() {
    return id;
  }

  /**
   * Gets the bluemixInstance.
   *
   * The IBM Cloud instance ID that identifies your Key Protect service instance.
   *
   * @return the bluemixInstance
   */
  public String bluemixInstance() {
    return bluemixInstance;
  }

  /**
   * Gets the keyRestoreBody.
   *
   * The base request for restore key action.
   *
   * @return the keyRestoreBody
   */
  public InputStream keyRestoreBody() {
    return keyRestoreBody;
  }

  /**
   * Gets the correlationId.
   *
   * The v4 UUID used to correlate and track transactions.
   *
   * @return the correlationId
   */
  public String correlationId() {
    return correlationId;
  }

  /**
   * Gets the prefer.
   *
   * Alters server behavior for POST or DELETE operations. A header with `return=minimal` causes the service to  return
   * only the key identifier, or metadata. A header containing `return=representation` returns both the key  material
   * and metadata in the response entity-body. If the key has been designated as a root key, the  system cannot return
   * the key material.
   *
   * **Note:** During POST operations, Key Protect may not immediately return the key material due to key generation
   * time. To retrieve the key material, you can perform a subsequent `GET /keys/{id}` request.
   *
   * @return the prefer
   */
  public String prefer() {
    return prefer;
  }
}

