package com.googlecode.mycontainer.js.junit;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.junit.runner.Description;
import org.junit.runner.notification.Failure;
import org.junit.runner.notification.RunNotifier;
import org.mozilla.javascript.Function;
import org.mozilla.javascript.NativeObject;

import com.googlecode.mycontainer.commons.rhino.RhinoBox;

public class JavaScriptTestCase {

    private URL jsUrl;

    private Description description;

    private List<JavaScriptTestUnit> units = new ArrayList<JavaScriptTestUnit>();

    private RhinoBox rhinoBox;

    private NativeObject testObject;

    private Function setUp;

    private Function tearDown;

    public JavaScriptTestCase(URL jsUrl) {
        this.jsUrl = jsUrl;
        init();
    }

    public boolean hasTestUnits() {
        return units.size() > 0;
    }

    private void init() {
        String path = jsUrl.getPath();
        rhinoBox = new RhinoBox();

        rhinoBox.enter();
        try {
            Object ret = rhinoBox.source(jsUrl);
            if (NativeObject.class.isInstance(ret)) {
                testObject = (NativeObject) ret;

                description = Description.createSuiteDescription(path);

                rhinoBox.getScope().set("testcase", ret);

                Object setUpElement = testObject.get("setUp", testObject);
                if (Function.class.isInstance(setUpElement)) {
                    setUp = (Function) setUpElement;
                }

                Object tearDownElement = testObject.get("tearDown", testObject);
                if (Function.class.isInstance(tearDownElement)) {
                    tearDown = (Function) tearDownElement;
                }

                for (Object obj : testObject.getAllIds()) {
                    if (String.class.isInstance(obj)) {
                        String elementName = (String) obj;
                        if (elementName.startsWith("test")) {
                            Object element = testObject.get(elementName, testObject);
                            if (Function.class.isInstance(element)) {
                                JavaScriptTestUnit unit = new JavaScriptTestUnit(path, elementName, (Function) element);

                                units.add(unit);
                                description.addChild(unit.getDescription());
                            }
                        }
                    }
                }
            }
        } finally {
            rhinoBox.exit();
        }
    }

    public Description getDescription() {
        return description;
    }

    public void run(RunNotifier notifier) {
        rhinoBox.enter();
        notifier.fireTestStarted(description);
        try {
            for (JavaScriptTestUnit unit : units) {
                try {
                    try {
                        setUp();
                        unit.run(rhinoBox, notifier);
                    } finally {
                        tearDown();
                    }
                } catch (Exception ex) {
                    // cause the setUp and tearDown fails
                    notifier.fireTestFailure(new Failure(description, ex));
                }
            }
        } finally {
            notifier.fireTestFinished(description);
            rhinoBox.exit();
        }
    }

    private void setUp() {
        if (setUp != null) {
            rhinoBox.invoke(getClass(), setUp);
        }
    }

    private void tearDown() {
        if (tearDown != null) {
            rhinoBox.invoke(getClass(), "testcase.tearDown");
        }
    }
}
