/*
 * Copyright 2010 Trustees of the University of Pennsylvania Licensed under the
 * Educational Community License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 *
 * http://www.osedu.org/licenses/ECL-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an "AS IS"
 * BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package metridoc.plugins.datasource

import javax.sql.DataSource
import metridoc.plugins.Plugin
import metridoc.plugins.datasource.providers.CommonsDataSourceProvider
import metridoc.plugins.datasource.providers.H2DataSourceProvider
import metridoc.plugins.datasource.providers.MySqlDataSourceProvider

@Plugin(category = "job")
class DataSourcePlugin {
    static dataSourceProviders = [h2: H2DataSourceProvider, mysql: MySqlDataSourceProvider]

    def static DataSource embeddedDataSource(Script self) {
        def url = "jdbc:h2:mem:metridocInMemory"
        use(DataSourcePlugin) {
            return self.dataSource(url: url)
        }
    }

    def static DataSource dataSource(Script self, LinkedHashMap args) {
        assert args.url || args.jdbcUrl: "A url must be provided to retrieve a datasource"
        DataSourcePluginUtils.validateAndFormatArgs(args)
        def provider = getProvider(args.url)

        if (provider) {
            return provider.getDataSource(args)
        }
    }

    private static getProvider(String url) {
        def m = (url =~ /jdbc:([^:]+)/)
        if (!m.lookingAt()) {
            throw new IllegalArgumentException("the passed jdbc url must be of the form jdbc:<provider>:<location>?<options>")
        }
        def providerName = m.group(1)
        def provider = dataSourceProviders[providerName]

        if (!provider) {
            provider = CommonsDataSourceProvider
        }

        return provider
    }
}