package com.github.xiaoluo.elasticjob.lite.autoconfigure;

import com.dangdang.ddframe.job.api.ElasticJob;
import com.dangdang.ddframe.job.api.dataflow.DataflowJob;
import com.dangdang.ddframe.job.api.simple.SimpleJob;
import com.dangdang.ddframe.job.config.JobCoreConfiguration;
import com.dangdang.ddframe.job.config.dataflow.DataflowJobConfiguration;
import com.dangdang.ddframe.job.config.simple.SimpleJobConfiguration;
import com.dangdang.ddframe.job.event.rdb.JobEventRdbConfiguration;
import com.dangdang.ddframe.job.executor.handler.JobProperties;
import com.dangdang.ddframe.job.lite.config.LiteJobConfiguration;
import com.dangdang.ddframe.job.lite.spring.api.SpringJobScheduler;
import com.dangdang.ddframe.job.reg.zookeeper.ZookeeperRegistryCenter;
import com.github.xiaoluo.elasticjob.lite.annotation.DataFlowElasticJob;
import com.github.xiaoluo.elasticjob.lite.annotation.SimpleElasticJob;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.beans.factory.support.ManagedList;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Configuration;

import javax.annotation.PostConstruct;
import javax.sql.DataSource;
import java.util.List;
import java.util.Map;

/**
 * Created with IntelliJ IDEA.
 *
 * @author: lee 2020/12/05
 */

@Configuration
@AutoConfigureAfter(RegistryCenterConfiguration.class)
@ConditionalOnBean(ZookeeperRegistryCenter.class)
public class ElasticJobAutoConfiguration {

    @Autowired
    private ApplicationContext applicationContext;

    @Autowired
    private ZookeeperRegistryCenter regCenter;

    @PostConstruct
    public void initElasticJob() {
        initElasticSimpleJob();
        initElasticDataflowJob();
    }

    public void initElasticSimpleJob() {
        Map<String, SimpleJob> map = applicationContext.getBeansOfType(SimpleJob.class);

        for (Map.Entry<String, SimpleJob> entry : map.entrySet()) {
            SimpleJob simpleJob = entry.getValue();
            SimpleElasticJob simpleElasticJob = simpleJob.getClass().getAnnotation(SimpleElasticJob.class);
            if (simpleElasticJob.disabled()) {
                continue;
            }
            String cron = simpleElasticJob.cron();
            String jobName = StringUtils.defaultIfBlank(simpleElasticJob.jobName(), simpleJob.getClass().getName());
            //job核心配置
            JobCoreConfiguration jobCoreConfiguration = JobCoreConfiguration.newBuilder(jobName, cron,
                    simpleElasticJob.shardingTotalCount())
                    .shardingItemParameters(simpleElasticJob.shardingItemParameters())
                    .failover(simpleElasticJob.failover())
                    .description(simpleElasticJob.description())
                    .jobParameter(simpleElasticJob.jobParameter())
                    .misfire(simpleElasticJob.misfire())
                    .jobProperties(JobProperties.JobPropertiesEnum.JOB_EXCEPTION_HANDLER.getKey(), simpleElasticJob.jobExceptionHandler())
                    .jobProperties(JobProperties.JobPropertiesEnum.EXECUTOR_SERVICE_HANDLER.getKey(),
                            simpleElasticJob.executorServiceHandler())
                    .build();

            //job类型配置
            SimpleJobConfiguration simpleJobConfiguration =
                    new SimpleJobConfiguration(jobCoreConfiguration,
                            simpleJob.getClass().getCanonicalName());
            //job根的配置
            LiteJobConfiguration liteJobConfiguration =
                    LiteJobConfiguration.newBuilder(simpleJobConfiguration)
                            .overwrite(simpleElasticJob.overwrite())
                            .disabled(simpleElasticJob.disabled())
                            .monitorPort(simpleElasticJob.monitorPort())
                            .monitorExecution(simpleElasticJob.monitorExecution())
                            .maxTimeDiffSeconds(simpleElasticJob.maxTimeDiffSeconds())
                            .jobShardingStrategyClass(simpleElasticJob.jobShardingStrategyClass())
                            .reconcileIntervalMinutes(simpleElasticJob.reconcileIntervalMinutes())
                            .build();
            List<BeanDefinition> elasticJobListeners = getTargetElasticJobListenersBySimpleElasticJob(simpleElasticJob);
            BeanDefinitionBuilder factory = BeanDefinitionBuilder.rootBeanDefinition(SpringJobScheduler.class);
            factory.setScope(BeanDefinition.SCOPE_PROTOTYPE);
            factory.addConstructorArgValue(simpleJob);
            factory.addConstructorArgValue(regCenter);
            factory.addConstructorArgValue(liteJobConfiguration);

            DefaultListableBeanFactory defaultListableBeanFactory =
                    (DefaultListableBeanFactory) applicationContext.getAutowireCapableBeanFactory();
            //是否开启事件追踪
            String dataSourceRef = simpleElasticJob.eventTraceRdbDataSource();
            if (StringUtils.isNotBlank(dataSourceRef)) {
//                throw new RuntimeException("not exist datasource [" + dataSourceRef + "] !");
//            }
//            DataSource dataSource = (DataSource) applicationContext.getBean(dataSourceRef);
//            JobEventRdbConfiguration jobEventRdbConfiguration = new JobEventRdbConfiguration(dataSource);  if (!applicationContext
//            .containsBean(dataSourceRef)) {

//                SpringJobScheduler jobScheduler = new SpringJobScheduler(simpleJob, regCenter, liteJobConfiguration,
//                        jobEventRdbConfiguration);
//                factory.addConstructorArgValue(jobEventRdbConfiguration);
                getJobEventRdb(factory, dataSourceRef);
                factory.addConstructorArgValue(elasticJobListeners);
                defaultListableBeanFactory.registerBeanDefinition(jobName + "SpringJobScheduler", factory.getBeanDefinition());
                SpringJobScheduler jobScheduler = (SpringJobScheduler) applicationContext.getBean(jobName + "SpringJobScheduler");
                jobScheduler.init();
            } else {
                factory.addConstructorArgValue(elasticJobListeners);
                //SpringJobScheduler jobScheduler = new SpringJobScheduler(simpleJob, regCenter, liteJobConfiguration);
                defaultListableBeanFactory.registerBeanDefinition(jobName + "SpringJobScheduler", factory.getBeanDefinition());
                SpringJobScheduler jobScheduler = (SpringJobScheduler) applicationContext.getBean(jobName + "SpringJobScheduler");
                jobScheduler.init();
            }
        }
    }

    public void initElasticDataflowJob() {
        Map<String, DataflowJob> map = applicationContext.getBeansOfType(DataflowJob.class);

        for (Map.Entry<String, DataflowJob> entry : map.entrySet()) {
            DataflowJob dataflowJob = entry.getValue();
            DataFlowElasticJob dataFlowElasticJob = dataflowJob.getClass().getAnnotation(DataFlowElasticJob.class);
            if (dataFlowElasticJob.disabled()) {
                continue;
            }
            String cron = dataFlowElasticJob.cron();
            String jobName = StringUtils.defaultIfBlank(dataFlowElasticJob.jobName(), dataflowJob.getClass().getName());
            //job核心配置
            JobCoreConfiguration jobCoreConfiguration = JobCoreConfiguration.newBuilder(jobName, cron,
                    dataFlowElasticJob.shardingTotalCount())
                    .shardingItemParameters(dataFlowElasticJob.shardingItemParameters())
                    .failover(dataFlowElasticJob.failover())
                    .description(dataFlowElasticJob.description())
                    .jobParameter(dataFlowElasticJob.jobParameter())
                    .misfire(dataFlowElasticJob.misfire())
                    .jobProperties(JobProperties.JobPropertiesEnum.JOB_EXCEPTION_HANDLER.getKey(), dataFlowElasticJob.jobExceptionHandler())
                    .jobProperties(JobProperties.JobPropertiesEnum.EXECUTOR_SERVICE_HANDLER.getKey(),
                            dataFlowElasticJob.executorServiceHandler())
                    .build();
            //job类型配置
            DataflowJobConfiguration dataflowJobConfiguration =
                    new DataflowJobConfiguration(jobCoreConfiguration, dataflowJob.getClass().getCanonicalName(),
                            dataFlowElasticJob.streamingProcess());
            //job根的配置
            LiteJobConfiguration liteJobConfiguration =
                    LiteJobConfiguration.newBuilder(dataflowJobConfiguration)
                            .overwrite(dataFlowElasticJob.overwrite())
                            .disabled(dataFlowElasticJob.disabled())
                            .monitorPort(dataFlowElasticJob.monitorPort())
                            .monitorExecution(dataFlowElasticJob.monitorExecution())
                            .maxTimeDiffSeconds(dataFlowElasticJob.maxTimeDiffSeconds())
                            .jobShardingStrategyClass(dataFlowElasticJob.jobShardingStrategyClass())
                            .reconcileIntervalMinutes(dataFlowElasticJob.reconcileIntervalMinutes())
                            .build();

            List<BeanDefinition> elasticJobListeners = getTargetElasticJobListenersByDataFlowElasticJob(dataFlowElasticJob);
            BeanDefinitionBuilder factory = BeanDefinitionBuilder.rootBeanDefinition(SpringJobScheduler.class);
            factory.setScope(BeanDefinition.SCOPE_PROTOTYPE);
            factory.addConstructorArgValue(dataflowJob);
            factory.addConstructorArgValue(regCenter);
            factory.addConstructorArgValue(liteJobConfiguration);

            DefaultListableBeanFactory defaultListableBeanFactory =
                    (DefaultListableBeanFactory) applicationContext.getAutowireCapableBeanFactory();
            //是否开启事件追踪
            String dataSourceRef = dataFlowElasticJob.eventTraceRdbDataSource();
            if (StringUtils.isNotBlank(dataSourceRef)) {
                getJobEventRdb(factory, dataSourceRef);
                factory.addConstructorArgValue(elasticJobListeners);
//                if (!applicationContext.containsBean(dataSourceRef)) {
//                    throw new RuntimeException("not exist datasource [" + dataSourceRef + "] !");
//                }
//                DataSource dataSource = (DataSource) applicationContext.getBean(dataSourceRef);
//                JobEventRdbConfiguration jobEventRdbConfiguration = new JobEventRdbConfiguration(dataSource);
//                factory.addConstructorArgValue(jobEventRdbConfiguration);
                defaultListableBeanFactory.registerBeanDefinition(jobName + "SpringJobScheduler", factory.getBeanDefinition());
                SpringJobScheduler jobScheduler = (SpringJobScheduler) applicationContext.getBean(jobName + "SpringJobScheduler");
//                SpringJobScheduler jobScheduler = new SpringJobScheduler(dataflowJob, regCenter, liteJobConfiguration,
//                        jobEventRdbConfiguration);
                jobScheduler.init();
            } else {
                factory.addConstructorArgValue(elasticJobListeners);
//                SpringJobScheduler jobScheduler = new SpringJobScheduler(dataflowJob, regCenter, liteJobConfiguration);
                defaultListableBeanFactory.registerBeanDefinition(jobName + "SpringJobScheduler", factory.getBeanDefinition());
                SpringJobScheduler jobScheduler = (SpringJobScheduler) applicationContext.getBean(jobName + "SpringJobScheduler");
                jobScheduler.init();
            }
        }
    }

    private void getJobEventRdb(BeanDefinitionBuilder factory, String dataSourceRef) {
        // 任务执行日志数据源，以名称获取
        if (org.springframework.util.StringUtils.hasText(dataSourceRef)) {
            BeanDefinitionBuilder rdbFactory = BeanDefinitionBuilder.rootBeanDefinition(JobEventRdbConfiguration.class);
            rdbFactory.addConstructorArgReference(dataSourceRef);
            factory.addConstructorArgValue(rdbFactory.getBeanDefinition());
        }
    }

    private List<BeanDefinition> getTargetElasticJobListenersByDataFlowElasticJob(DataFlowElasticJob dataFlowElasticJob) {
        List<BeanDefinition> result = new ManagedList<BeanDefinition>(2);
        String listeners = dataFlowElasticJob.listener();
        if (org.springframework.util.StringUtils.hasText(listeners)) {
            BeanDefinitionBuilder factory = BeanDefinitionBuilder.rootBeanDefinition(listeners);
            factory.setScope(BeanDefinition.SCOPE_PROTOTYPE);
            result.add(factory.getBeanDefinition());
        }
        String distributedListeners = dataFlowElasticJob.distributedListener();
        long startedTimeoutMilliseconds = dataFlowElasticJob.startedTimeoutMilliseconds();
        long completedTimeoutMilliseconds = dataFlowElasticJob.completedTimeoutMilliseconds();
        if (org.springframework.util.StringUtils.hasText(distributedListeners)) {
            BeanDefinitionBuilder factory = BeanDefinitionBuilder.rootBeanDefinition(distributedListeners);
            factory.setScope(BeanDefinition.SCOPE_PROTOTYPE);
            factory.addConstructorArgValue(startedTimeoutMilliseconds);
            factory.addConstructorArgValue(completedTimeoutMilliseconds);
            result.add(factory.getBeanDefinition());
        }
        return result;
    }

    private List<BeanDefinition> getTargetElasticJobListenersBySimpleElasticJob(SimpleElasticJob simpleElasticJob) {
        List<BeanDefinition> result = new ManagedList<BeanDefinition>(2);
        String listeners = simpleElasticJob.listener();
        if (org.springframework.util.StringUtils.hasText(listeners)) {
            BeanDefinitionBuilder factory = BeanDefinitionBuilder.rootBeanDefinition(listeners);
            factory.setScope(BeanDefinition.SCOPE_PROTOTYPE);
            result.add(factory.getBeanDefinition());
        }
        String distributedListeners = simpleElasticJob.distributedListener();
        long startedTimeoutMilliseconds = simpleElasticJob.startedTimeoutMilliseconds();
        long completedTimeoutMilliseconds = simpleElasticJob.completedTimeoutMilliseconds();
        if (org.springframework.util.StringUtils.hasText(distributedListeners)) {
            BeanDefinitionBuilder factory = BeanDefinitionBuilder.rootBeanDefinition(distributedListeners);
            factory.setScope(BeanDefinition.SCOPE_PROTOTYPE);
            factory.addConstructorArgValue(startedTimeoutMilliseconds);
            factory.addConstructorArgValue(completedTimeoutMilliseconds);
            result.add(factory.getBeanDefinition());
        }
        return result;
    }
}
