package com.github.xiaoluo.elasticjob.lite.autoconfigure;

import com.dangdang.ddframe.job.api.dataflow.DataflowJob;
import com.dangdang.ddframe.job.api.simple.SimpleJob;
import com.dangdang.ddframe.job.config.JobCoreConfiguration;
import com.dangdang.ddframe.job.config.dataflow.DataflowJobConfiguration;
import com.dangdang.ddframe.job.config.simple.SimpleJobConfiguration;
import com.dangdang.ddframe.job.event.rdb.JobEventRdbConfiguration;
import com.dangdang.ddframe.job.lite.config.LiteJobConfiguration;
import com.dangdang.ddframe.job.lite.spring.api.SpringJobScheduler;
import com.dangdang.ddframe.job.reg.zookeeper.ZookeeperRegistryCenter;
import com.github.xiaoluo.elasticjob.lite.annotation.DataFlowElasticJob;
import com.github.xiaoluo.elasticjob.lite.annotation.SimpleElasticJob;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Configuration;

import javax.annotation.PostConstruct;
import javax.sql.DataSource;
import java.util.Map;

/**
 * Created with IntelliJ IDEA.
 *
 * @author: lee 2020/12/05
 */

@Configuration
@ConditionalOnExpression("'${elasticjob.zookeeper.server-lists}'.length() > 0")
@AutoConfigureAfter(RegistryCenterConfiguration.class)
public class ElasticJobAutoConfiguration {

    @Autowired
    private ApplicationContext applicationContext;

    @Autowired
    private ZookeeperRegistryCenter regCenter;

    @PostConstruct
    public void initElasticJob() {
        initElasticSimpleJob();
        initElasticDataflowJob();
    }

    public void initElasticSimpleJob() {
        Map<String, SimpleJob> map = applicationContext.getBeansOfType(SimpleJob.class);

        for (Map.Entry<String, SimpleJob> entry : map.entrySet()) {
            SimpleJob simpleJob = entry.getValue();
            SimpleElasticJob simpleElasticJob = simpleJob.getClass().getAnnotation(SimpleElasticJob.class);
            if (simpleElasticJob.disabled()) {
                continue;
            }
            String cron = simpleElasticJob.cron();
            String jobName = StringUtils.defaultIfBlank(simpleElasticJob.jobName(), simpleJob.getClass().getName());
            //job核心配置
            JobCoreConfiguration jobCoreConfiguration = JobCoreConfiguration.newBuilder(jobName, cron,
                    simpleElasticJob.shardingTotalCount()).shardingItemParameters(simpleElasticJob.shardingItemParameters()).build();
            //job类型配置
            SimpleJobConfiguration simpleJobConfiguration =
                    new SimpleJobConfiguration(jobCoreConfiguration,
                            simpleJob.getClass().getCanonicalName());
            LiteJobConfiguration liteJobConfiguration = getLiteJobConfiguration(simpleElasticJob, simpleJobConfiguration);
            //是否开启事件追踪
            String dataSourceRef = simpleElasticJob.dataSource();
            if (StringUtils.isNotBlank(dataSourceRef)) {
                if (!applicationContext.containsBean(dataSourceRef)) {
                    throw new RuntimeException("not exist datasource [" + dataSourceRef + "] !");
                }

                DataSource dataSource = (DataSource) applicationContext.getBean(dataSourceRef);
                JobEventRdbConfiguration jobEventRdbConfiguration = new JobEventRdbConfiguration(dataSource);
                SpringJobScheduler jobScheduler = new SpringJobScheduler(simpleJob, regCenter, liteJobConfiguration,
                        jobEventRdbConfiguration);
                jobScheduler.init();
            } else {
                SpringJobScheduler jobScheduler = new SpringJobScheduler(simpleJob, regCenter, liteJobConfiguration);
                jobScheduler.init();
            }
        }
    }

    private LiteJobConfiguration getLiteJobConfiguration(SimpleElasticJob simpleElasticJob, SimpleJobConfiguration simpleJobConfiguration) {
        //job根的配置
        return LiteJobConfiguration.newBuilder(simpleJobConfiguration).overwrite(simpleElasticJob.overwrite()).build();
    }

    public void initElasticDataflowJob() {
        Map<String, DataflowJob> map = applicationContext.getBeansOfType(DataflowJob.class);

        for (Map.Entry<String, DataflowJob> entry : map.entrySet()) {
            DataflowJob dataflowJob = entry.getValue();
            DataFlowElasticJob dataFlowElasticJob = dataflowJob.getClass().getAnnotation(DataFlowElasticJob.class);
            if (dataFlowElasticJob.disabled()) {
                continue;
            }
            String cron = dataFlowElasticJob.cron();
            String jobName = StringUtils.defaultIfBlank(dataFlowElasticJob.jobName(), dataflowJob.getClass().getName());
            //job核心配置
            JobCoreConfiguration jobCoreConfiguration = JobCoreConfiguration.newBuilder(jobName, cron,
                    dataFlowElasticJob.shardingTotalCount()).shardingItemParameters(dataFlowElasticJob.shardingItemParameters()).build();
            //job类型配置
            DataflowJobConfiguration dataflowJobConfiguration =
                    new DataflowJobConfiguration(jobCoreConfiguration, dataflowJob.getClass().getCanonicalName(),
                            dataFlowElasticJob.streamingProcess());
            //job根的配置
            LiteJobConfiguration liteJobConfiguration =
                    LiteJobConfiguration.newBuilder(dataflowJobConfiguration).overwrite(dataFlowElasticJob.overwrite()).build();
            //是否开启事件追踪
            String dataSourceRef = dataFlowElasticJob.dataSource();
            if (StringUtils.isNotBlank(dataSourceRef)) {

                if (!applicationContext.containsBean(dataSourceRef)) {
                    throw new RuntimeException("not exist datasource [" + dataSourceRef + "] !");
                }

                DataSource dataSource = (DataSource) applicationContext.getBean(dataSourceRef);
                JobEventRdbConfiguration jobEventRdbConfiguration = new JobEventRdbConfiguration(dataSource);
                SpringJobScheduler jobScheduler = new SpringJobScheduler(dataflowJob, regCenter, liteJobConfiguration,
                        jobEventRdbConfiguration);
                jobScheduler.init();
            } else {
                SpringJobScheduler jobScheduler = new SpringJobScheduler(dataflowJob, regCenter, liteJobConfiguration);
                jobScheduler.init();
            }
        }
    }
}
