package com.github.visionaryappdev.auditgateway.filter;

import brave.Span;
import brave.Tracer;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.reactivestreams.Publisher;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.cloud.gateway.filter.GatewayFilterChain;
import org.springframework.cloud.gateway.filter.GlobalFilter;
import org.springframework.cloud.gateway.filter.NettyWriteResponseFilter;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferFactory;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpResponse;
import org.springframework.http.server.reactive.ServerHttpResponseDecorator;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;

//@EnableAutoConfiguration
//@Configuration
@Component
public class GatewayResponseFilter implements GlobalFilter, Ordered {

    @Autowired
    private Tracer tracer;

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, GatewayFilterChain chain) {
        ServerHttpResponse originalResponse = exchange.getResponse();
        DataBufferFactory bufferFactory = originalResponse.bufferFactory();

        ServerHttpResponseDecorator decoratedResponse = new ServerHttpResponseDecorator(originalResponse) {
            @Override
            public Mono<Void> writeWith(Publisher<? extends DataBuffer> body) {
                if (body instanceof Flux) {
                    Flux<? extends DataBuffer> fluxBody = (Flux<? extends DataBuffer>) body;
                    return super.writeWith(fluxBody.map(dataBuffer -> {
                        Span span = tracer.nextSpan().name("response");
                        // probably should reuse buffers
                        byte[] responseContent = new byte[dataBuffer.readableByteCount()];
                        dataBuffer.read(responseContent);
                        String uppedContent = new String(responseContent, StandardCharsets.UTF_8);


                        span.tag("response-body", prepareResponseBody(exchange, uppedContent));
                        span.tag("headers", originalResponse.getHeaders().toString());
                        span.tag("status", String.valueOf(originalResponse.getStatusCode()));


                        span.finish();
                        return bufferFactory.wrap(responseContent); // return byte of modified data here
                    }));
                }
                return super.writeWith(body);
            }
        };

        exchange.getResponse().getHeaders().add("correlation-id", tracer.currentSpan().context().traceIdString());
        return chain.filter(exchange.mutate().response(decoratedResponse).build()); // replace response with decorator that we just decorated
    }


    @Override
    public int getOrder() {
        /// Order must b4 netty response to client or netty already response and we modify
        return NettyWriteResponseFilter.WRITE_RESPONSE_FILTER_ORDER - 1;
    }


    private String prepareResponseBody(ServerWebExchange exchange, String originalContent) {
        MediaType contentType = exchange.getRequest().getHeaders().getContentType();

        if (contentType != null && contentType.includes(MediaType.APPLICATION_JSON)) {
            ObjectMapper map = new ObjectMapper();
            TypeReference<HashMap<String, Object>> typeRef = new TypeReference<>() {
            };

            try {
                HashMap<String, Object> responseBody = map.readValue(originalContent, typeRef);
                responseBody.forEach((key, val) -> {
                    if (key.toLowerCase().contains("refresh_token") || key.toLowerCase().contains("access_token")) {
                        responseBody.put(key, "***");
                    }
                });

                return responseBody.toString();
            } catch (JsonProcessingException e) {
                e.printStackTrace();
            }
        } else if (contentType != null
                && (contentType.includes(MediaType.APPLICATION_XML) || contentType.includes(MediaType.TEXT_HTML) || contentType.includes(MediaType.TEXT_PLAIN)))
            return originalContent;


        return "";
    }
}

