package com.github.visionaryappdev.auditgateway.filter;

import brave.Span;
import brave.Tracer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.gateway.filter.GlobalFilter;
import org.springframework.cloud.gateway.filter.NettyWriteResponseFilter;
import org.springframework.cloud.gateway.filter.factory.rewrite.ModifyRequestBodyGatewayFilterFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.core.annotation.Order;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import java.util.*;

/***
 * Extends ModifyRequestBodyGatewayFilterFactory and
 * \@Override
 * GatewayFilter apply()
 * and add ClassName to
 * default-filters: - ClassName
 *
 *
 * or
 *  \@Bean
 *  GlobalFilter apply()
 */
//@EnableAutoConfiguration
//@Configuration
@Component
public class GatewayRequestFilter {//extends ModifyRequestBodyGatewayFilterFactory {

    @Autowired
    private Tracer tracer;


    //    @Override
//    public GatewayFilter apply(Config config) {
    @Order(NettyWriteResponseFilter.WRITE_RESPONSE_FILTER_ORDER - 1)
    @Bean
    public GlobalFilter apply() {
        return (exchange, chain) -> {
            MediaType contentType = exchange.getRequest().getHeaders().getContentType();
            ModifyRequestBodyGatewayFilterFactory.Config modifyRequestConfig = new ModifyRequestBodyGatewayFilterFactory.Config();


            if (contentType == null) {
                return chain.filter(exchange);
            } else if (contentType.includes(MediaType.APPLICATION_JSON)) {
                modifyRequestConfig.setRewriteFunction(Object.class, Object.class, (exchange1, originalRequestBody) -> {
                    auditLog(exchange1, originalRequestBody);
                    return Mono.just(originalRequestBody);
                });
            } else if (contentType.includes(MediaType.TEXT_PLAIN) || contentType.includes(MediaType.APPLICATION_XML)) {
                modifyRequestConfig.setRewriteFunction(String.class, String.class, (exchange1, originalRequestBody) -> {
                    auditLog(exchange1, originalRequestBody);
                    return Mono.just(originalRequestBody);
                });
            } else if (contentType.includes(MediaType.MULTIPART_FORM_DATA)) {
                modifyRequestConfig.setRewriteFunction(String.class, String.class, (exchange1, originalRequestBody) -> {
                    auditLog(exchange1, parseFormData(originalRequestBody));
                    return Mono.just(originalRequestBody);
                    // This method we could read request content but it got stuck and hang for few seconds so it's not a way
//                return exchange.getMultipartData().flatMap(originalRequestBody -> {
//                    auditLog(exchange, originalRequestBody);
//                    return chain.filter(exchange);
                });
            } else if (contentType.includes(MediaType.APPLICATION_FORM_URLENCODED)) {
                modifyRequestConfig.setRewriteFunction(MultiValueMap.class, MultiValueMap.class, (exchange1, originalRequestBody) -> {
                    auditLog(exchange1, originalRequestBody);
                    return Mono.just(originalRequestBody);
                });
                // This method we could read request content but it got stuck and hang for few seconds so it's not a way
//                return exchange.getFormData().flatMap(originalRequestBody -> {
//                    auditLog(exchange, originalRequestBody);
//                    return chain.filter(exchange);
//                });
            } else {
                // https://github.com/spring-cloud/spring-cloud-gateway/issues/509#issuecomment-415008272
                return chain.filter(exchange);
            }

            return new ModifyRequestBodyGatewayFilterFactory().apply(modifyRequestConfig).filter(exchange, chain);
        };
    }

    private Map<String, Map<String, String>> parseFormData(String originalRequestBody) {
        List<String> formData = new ArrayList<>(Arrays.asList(originalRequestBody.split("----------------------------")));
        /// Remove delimiter if data is not null
        if (!formData.isEmpty()) {
            formData.remove(0);
            formData.remove(formData.size() - 1);
        }


        Map<String, Map<String, String>> result = new HashMap<>();
        formData.forEach(inData -> {
            Map<String, String> eachField = new HashMap<>();


            /// Remove Hash Line
            int ind = inData.indexOf('\n');
            String procData = inData.substring(ind + 1);


            /// Field Name
            ind = procData.indexOf('\n');
            List<String> firstLine = new ArrayList<>(Arrays.asList(procData.substring(0, ind + 1).replace("Content-Disposition: form-data; name=\"", "").trim().split("; ")));
            String fieldName = firstLine.get(0).substring(0, firstLine.get(0).length() - 1);

            /// File Name
            if (firstLine.size() > 1) {
                String fileName = firstLine.get(1).trim().replace("filename=\"", "");
                int fileNameInd = fileName.lastIndexOf('\"');
                fileName = fileName.substring(0, fileNameInd - 1);

                eachField.put("file name", fileName);
            }


            /// Value
            ind = procData.indexOf('\n');
            procData = procData.substring(ind + 1);
            ind = procData.indexOf('\n');
            String contentTyp = procData.substring(0, ind + 1).trim();

            if (contentTyp.equals("")) {
                eachField.put("value", procData.trim());
            }


            /// Multiple file
            if (result.get(fieldName) != null) {
                result.put(fieldName + "--" + (new Random()).nextInt(512), eachField);
            } else {
                result.put(fieldName, eachField);
            }
        });

        return result;
    }

    private void auditLog(ServerWebExchange exchange, Object originalRequestBody) {
        Span span = tracer.nextSpan().name("request");

        try {
            Map<String, Object> headers = prepareHeaders(exchange);
            String queryParam = exchange.getRequest().getQueryParams().toString();
            MediaType contentType = exchange.getRequest().getHeaders().getContentType();
            List<String> realIp = exchange.getRequest().getHeaders().get("X-Real-IP");
            List<String> forwardIp = exchange.getRequest().getHeaders().get("X-Forwarded-For");


            if (contentType != null)
                span.tag("content-type", contentType.toString());
            if (realIp != null && !realIp.isEmpty())
                span.tag("X-Real-IP", realIp.get(0));
            if (forwardIp != null && !forwardIp.isEmpty())
                span.tag("X-Forwarded-For", forwardIp.get(0));


            span.tag("headers", headers.toString());
            span.tag("query-param", queryParam);
            span.tag("request-body", prepareRequestBody(contentType, originalRequestBody));
        } catch (Exception e) {
            span.error(e);
        } finally {
            span.finish();
        }


//        In case of decorator, then response the below
//        return Mono.just(originalRequestBody);
    }


    @SuppressWarnings("unchecked")
    private String prepareRequestBody(MediaType contentType, Object originalRequestBody) {
        if (contentType == null) {
            return originalRequestBody.toString();
        } else if (contentType.includes(MediaType.APPLICATION_JSON)) {
            Map<String, Object> body = new LinkedHashMap<>((LinkedHashMap<String, Object>) originalRequestBody);
            body.forEach((key, value) -> {
                if (key.toLowerCase().contains("client_secret") || key.toLowerCase().contains("password") || key.toLowerCase().contains("refresh_token")) {
                    body.put(key, "***");
                }
            });

            return body.toString();
        } else if (contentType.includes(MediaType.MULTIPART_FORM_DATA)) {
            Map<String, Map<String, String>> body = new HashMap<>((Map<String, Map<String, String>>) originalRequestBody);
            body.forEach((key, value) -> {
                value.forEach((key1, value1) -> {
                    if (key1.toLowerCase().contains("client_secret") || key1.toLowerCase().contains("password") || key1.toLowerCase().contains("refresh_token")) {
                        body.get(key).put(key1, "***");
                    }
                });
            });

            return body.toString();
        } else if (contentType.includes(MediaType.APPLICATION_FORM_URLENCODED)) {
            MultiValueMap<String, Object> body = new LinkedMultiValueMap<>((LinkedMultiValueMap<String, Object>) originalRequestBody);
            body.forEach((key, value) -> {
                if (key.toLowerCase().contains("client_secret") || key.toLowerCase().contains("password") || key.toLowerCase().contains("refresh_token")) {
                    body.put(key, List.of("***"));
                }
            });

            return body.toString();
        } else // (exchange.getRequest().getHeaders().getContentType().includes(MediaType.TEXT_PLAIN))
            return originalRequestBody.toString();
    }


    private Map<String, Object> prepareHeaders(ServerWebExchange exchange) {
        /// Mask sensitive headers information
        Map<String, Object> headers = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        headers.putAll(exchange.getRequest().getHeaders());
        if (headers.containsKey("authorization")) {
            headers.put("authorization", "***");
        }

        return headers;
    }
}
