package com.unbound.provider.kmip.attribute;


import com.unbound.provider.kmip.KMIP;
import com.unbound.provider.kmip.KMIPConvertException;
import com.unbound.provider.kmip.KMIPConverter;

import java.util.HashMap;
import java.util.Map;

/**
 * Created by valery.osheter on 18-Nov-15.
 */
public abstract class Attribute
{
  public int tag = 0;
  public Integer index = null;

  public Attribute()
  {

  }

  public Attribute(int _tag)
  {
    tag = _tag;
  }


  public static class AttributeDef
  {
    public int tag;
    public String name;

    AttributeDef(int tag, String name)
    {
      this.tag = tag;
      this.name = name;
    }

    Attribute create()
    {
      return null;
    }
  }

  private static class AttributeDefClass extends AttributeDef
  {
    Class<? extends Attribute> clazz;

    AttributeDefClass(int tag, String name, Class<? extends Attribute> clazz)
    {
      super(tag, name);
      this.clazz = clazz;
    }

    Attribute create()
    {
      try
      {
        return clazz.newInstance();
      }
      catch (InstantiationException e)
      {
        return null;
      }
      catch (IllegalAccessException e)
      {
        return null;
      }
    }
  }

  private static class AttributeDefInt extends AttributeDef
  {
    AttributeDefInt(int tag, String name)
    {
      super(tag, name);
    }

    Attribute create()
    {
      return new IntAttribute(tag);
    }
  }

  private static class AttributeDefBool extends AttributeDef
  {
    AttributeDefBool(int tag, String name)
    {
      super(tag, name);
    }

    Attribute create()
    {
      return new BoolAttribute(tag);
    }
  }

  private static class AttributeDefDate extends AttributeDef
  {
    AttributeDefDate(int tag, String name)
    {
      super(tag, name);
    }

    Attribute create()
    {
      return new DateAttribute(tag);
    }
  }

  private static class AttributeDefInt64 extends AttributeDef
  {
    AttributeDefInt64(int tag, String name)
    {
      super(tag, name);
    }

    Attribute create()
    {
      return new Int64Attribute(tag);
    }
  }

  private static class AttributeDefBigInt extends AttributeDef
  {
    AttributeDefBigInt(int tag, String name)
    {
      super(tag, name);
    }

    Attribute create()
    {
      return new BigNumAttribute(tag);
    }
  }

  private static class AttributeDefText extends AttributeDef
  {
    AttributeDefText(int tag, String name)
    {
      super(tag, name);
    }

    Attribute create()
    {
      return new TextAttribute(tag);
    }
  }

  private static class AttributeDefBytes extends AttributeDef
  {
    AttributeDefBytes(int tag, String name)
    {
      super(tag, name);
    }

    Attribute create()
    {
      return new BytesAttribute(tag);
    }
  }

  private static class AttributeDefEnum extends AttributeDef
  {
    AttributeDefEnum(int tag, String name)
    {
      super(tag, name);
    }

    Attribute create()
    {
      return new EnumAttribute(tag);
    }
  }

  public String getName()
  {
    AttributeDef def = tags.get(tag);
    if (def == null) return null;
    return def.name;
  }

  private static final Map<Integer, AttributeDef> tags = new HashMap<Integer, AttributeDef>();
  private static final Map<String, AttributeDef> names = new HashMap<String, AttributeDef>();

  private static void register(AttributeDef def)
  {
    tags.put(def.tag, def);
    names.put(def.name, def);
  }

  static
  {
    register(new AttributeDefClass(KMIP.Tag.Name, "Name", Name.class));
    register(new AttributeDefClass(KMIP.Tag.AlternativeName, "Alternative Name", AlternativeName.class));
    register(new AttributeDefText(KMIP.Tag.UniqueIdentifier, "Unique Identifier"));
    register(new AttributeDefEnum(KMIP.Tag.CryptographicAlgorithm, "Cryptographic Algorithm"));
    register(new AttributeDefInt(KMIP.Tag.CryptographicLength, "Cryptographic Length"));
    register(new AttributeDefEnum(KMIP.Tag.ObjectType, "Object Type"));
    register(new AttributeDefClass(KMIP.Tag.CryptographicParameters, "Cryptographic Parameters", CryptoParams.class));
    register(new AttributeDefClass(KMIP.Tag.CryptographicDomainParameters, "Cryptographic Domain Parameters", CryptoDomainParams.class));
    register(new AttributeDefInt(KMIP.Tag.CertificateLength, "Certificate Length"));
    register(new AttributeDefEnum(KMIP.Tag.CertificateType, "Certificate Type"));
    register(new AttributeDefClass(KMIP.Tag.X509CertificateIdentifier, "X.509 Certificate Identifier", X509CertificateID.class));
    register(new AttributeDefClass(KMIP.Tag.X509CertificateSubject, "X.509 Certificate Subject", X509CertificateSubject.class));
    register(new AttributeDefClass(KMIP.Tag.X509CertificateIssuer, "X.509 Certificate Issuer", X509CertificateIssuer.class));
    register(new AttributeDefClass(KMIP.Tag.CertificateIdentifier, "Certificate Identifier", CertificateID.class));
    register(new AttributeDefClass(KMIP.Tag.CertificateSubject, "Certificate Subject", CertificateSubject.class));
    register(new AttributeDefClass(KMIP.Tag.CertificateIssuer, "Certificate Issuer", CertificateIssuer.class));
    register(new AttributeDefEnum(KMIP.Tag.DigitalSignatureAlgorithm, "Digital Signature Algorithm"));
    register(new AttributeDefClass(KMIP.Tag.Digest, "Digest", Digest.class));
    register(new AttributeDefText(KMIP.Tag.OperationPolicyName, "Operation Policy Name"));
    register(new AttributeDefInt(KMIP.Tag.CryptographicUsageMask, "Cryptographic Usage Mask"));
    register(new AttributeDefEnum(KMIP.Tag.State, "State"));
    register(new AttributeDefDate(KMIP.Tag.InitialDate, "Initial Date"));
    register(new AttributeDefDate(KMIP.Tag.ActivationDate, "Activation Date"));
    register(new AttributeDefDate(KMIP.Tag.DeactivationDate, "Deactivation Date"));
    register(new AttributeDefDate(KMIP.Tag.CompromiseOccurrenceDate, "Compromise Occurrence Date"));
    register(new AttributeDefDate(KMIP.Tag.CompromiseDate, "Compromise Date"));
    register(new AttributeDefDate(KMIP.Tag.LastChangeDate, "Last Change Date"));
    register(new AttributeDefClass(KMIP.Tag.RevocationReason, "Revocation Reason", RevocationReason.class));
    register(new AttributeDefText(KMIP.Tag.ObjectGroup, "Object Group"));
    register(new AttributeDefClass(KMIP.Tag.ApplicationSpecificInformation, "Application Specific Information", AppInfo.class));
    register(new AttributeDefText(KMIP.Tag.ContactInformation, "Contact Information"));

    register(new AttributeDefText(KMIP.Tag.CertificateSubjectAlternativeName, "Certificate Subject Alternative Name"));
    register(new AttributeDefDate(KMIP.Tag.OriginalCreationDate, "Original Creation Date"));

    register(new AttributeDefClass(KMIP.Tag.RNGParameters, "Random Number Generator", RNGParams.class));

    register(new AttributeDefDate(KMIP.Tag.ProcessStartDate, "Process Start Date"));
    register(new AttributeDefDate(KMIP.Tag.ProtectStopDate, "Protect Stop Date"));
    register(new AttributeDefClass(KMIP.Tag.UsageLimits, "Usage Limits", UsageLimits.class));
    register(new AttributeDefClass(KMIP.Tag.KeyValueLocation, "Key Value Location", KeyValueLocation.class));
    register(new AttributeDefDate(KMIP.Tag.LastChangeDate, "Last Change Date"));

    register(new AttributeDefBool(KMIP.Tag.Sensitive, "Sensitive"));
    register(new AttributeDefBool(KMIP.Tag.AlwaysSensitive, "AlwaysSensitive"));
    register(new AttributeDefBool(KMIP.Tag.Extractable, "Extractable"));
    register(new AttributeDefBool(KMIP.Tag.NeverExtractable, "NeverExtractable"));

    register(new AttributeDefEnum(KMIP.Tag.RecommendedCurve, "RecommendedCurve"));
    register(new AttributeDefEnum(KMIP.Tag.SecretDataType, "Secret Data Type"));

    register(new AttributeDefInt64(KMIP.Tag.DyDYCKA_UID, "x-DYCKA_UID"));
    register(new AttributeDefBytes(KMIP.Tag.DyDYCKA_PASS_EC_POINT, "x-DYCKA_PASS_EC_POINT"));
    register(new AttributeDefBytes(KMIP.Tag.DyDYCKA_DATA_EC_POINT, "x-DYCKA_DATA_EC_POINT"));
    register(new AttributeDefBytes(KMIP.Tag.DyDYCKA_LIMA_PUB_KEY, "x-DYCKA_LIMA_PUB_KEY"));
    register(new AttributeDefBigInt(KMIP.Tag.DyDYCKA_ECDSA_ENC_PUB_KEY, "x-DYCKA_ECDSA_ENC_PUB_KEY"));
    register(new AttributeDefBool(KMIP.Tag.DyDYCKA_ERASE_KEY_MATERIAL, "x-DYCKA_ERASE_KEY_MATERIAL"));
    register(new AttributeDefBytes(KMIP.Tag.DyDYCKA_EDDSA_PUB_KEY, "x-DYCKA_EDDSA_PUB_KEY"));

    register(new AttributeDefEnum(KMIP.Tag.CKA_CLASS, "x-CKA_CLASS"));
    register(new AttributeDefBool(KMIP.Tag.CKA_TOKEN, "x-CKA_TOKEN"));
    register(new AttributeDefBool(KMIP.Tag.CKA_PRIVATE, "x-CKA_PRIVATE"));
    register(new AttributeDefText(KMIP.Tag.CKA_LABEL, "x-CKA_LABEL"));
    register(new AttributeDefBool(KMIP.Tag.DyDoActivate, "x-DY_DO_ACTIVATE"));
    register(new AttributeDefBool(KMIP.Tag.DyKeyOnly, "x-DY_KEY_ONLY"));
    register(new AttributeDefBool(KMIP.Tag.DyTestMode, "x-DY_TEST_MODE"));
    register(new AttributeDefText(KMIP.Tag.DyTemplate, "x-DY_CLIENT_TEMPLATE"));

    register(new AttributeDefText(KMIP.Tag.CKA_APPLICATION, "x-CKA_APPLICATION"));
    register(new AttributeDefBytes(KMIP.Tag.CKA_VALUE, "x-CKA_VALUE"));
    register(new AttributeDefEnum(KMIP.Tag.CKA_CERTIFICATE_TYPE, "x-CKA_CERTIFICATE_TYPE"));
    register(new AttributeDefBytes(KMIP.Tag.CKA_ISSUER, "x-CKA_ISSUER"));
    register(new AttributeDefBytes(KMIP.Tag.CKA_SERIAL_NUMBER, "x-CKA_SERIAL_NUMBER"));
    register(new AttributeDefEnum(KMIP.Tag.CKA_CERTIFICATE_CATEGORY, "x-CKA_CERTIFICATE_CATEGORY"));
    register(new AttributeDefText(KMIP.Tag.CKA_URL, "x-CKA_URL"));
    register(new AttributeDefBytes(KMIP.Tag.CKA_OBJECT_ID, "x-CKA_OBJECT_ID"));
    register(new AttributeDefBytes(KMIP.Tag.CKA_CHECK_VALUE, "x-CKA_CHECK_VALUE"));
    register(new AttributeDefEnum(KMIP.Tag.CKA_KEY_GEN_MECHANISM, "x-CKA_KEY_GEN_MECHANISM"));
    register(new AttributeDefEnum(KMIP.Tag.CKA_KEY_TYPE, "x-CKA_KEY_TYPE"));
    register(new AttributeDefBytes(KMIP.Tag.CKA_SUBJECT, "x-CKA_SUBJECT"));
    register(new AttributeDefBytes(KMIP.Tag.CKA_ID, "x-CKA_ID"));
    register(new AttributeDefBool(KMIP.Tag.CKA_SENSITIVE, "x-CKA_SENSITIVE"));
    register(new AttributeDefBool(KMIP.Tag.CKA_ENCRYPT, "x-CKA_ENCRYPT"));
    register(new AttributeDefBool(KMIP.Tag.CKA_DECRYPT, "x-CKA_DECRYPT"));
    register(new AttributeDefBool(KMIP.Tag.CKA_WRAP, "x-CKA_WRAP"));
    register(new AttributeDefBool(KMIP.Tag.CKA_UNWRAP, "x-CKA_UNWRAP"));
    register(new AttributeDefBool(KMIP.Tag.CKA_SIGN, "x-CKA_SIGN"));
    register(new AttributeDefBool(KMIP.Tag.CKA_SIGN_RECOVER, "x-CKA_SIGN_RECOVER"));
    register(new AttributeDefBool(KMIP.Tag.CKA_VERIFY, "x-CKA_VERIFY"));
    register(new AttributeDefBool(KMIP.Tag.CKA_VERIFY_RECOVER, "x-CKA_VERIFY_RECOVER"));
    register(new AttributeDefBool(KMIP.Tag.CKA_DERIVE, "x-CKA_DERIVE"));
    register(new AttributeDefDate(KMIP.Tag.CKA_START_DATE, "x-CKA_START_DATE"));
    register(new AttributeDefDate(KMIP.Tag.CKA_END_DATE, "x-CKA_END_DATE"));
    register(new AttributeDefBigInt(KMIP.Tag.CKA_MODULUS, "x-CKA_MODULUS"));
    register(new AttributeDefInt(KMIP.Tag.CKA_MODULUS_BITS, "x-CKA_MODULUS_BITS"));
    register(new AttributeDefBigInt(KMIP.Tag.CKA_PUBLIC_EXPONENT, "x-CKA_PUBLIC_EXPONENT"));
    //register(new AttributeDefBigInt (KMIP.Tag.CKA_PRIVATE_EXPONENT,     "x-CKA_PRIVATE_EXPONENT"));
    //register(new AttributeDefBigInt (KMIP.Tag.CKA_PRIME_1,              "x-CKA_PRIME_1"));
    //register(new AttributeDefBigInt (KMIP.Tag.CKA_PRIME_2,              "x-CKA_PRIME_2"));
    //register(new AttributeDefBigInt (KMIP.Tag.CKA_EXPONENT_1,           "x-CKA_EXPONENT_1"));
    //register(new AttributeDefBigInt (KMIP.Tag.CKA_EXPONENT_2,           "x-CKA_EXPONENT_2"));
    register(new AttributeDefBigInt(KMIP.Tag.CKA_COEFFICIENT, "x-CKA_COEFFICIENT"));
    register(new AttributeDefInt(KMIP.Tag.CKA_VALUE_LEN, "x-CKA_VALUE_LEN"));
    register(new AttributeDefBool(KMIP.Tag.CKA_EXTRACTABLE, "x-CKA_EXTRACTABLE"));
    register(new AttributeDefBool(KMIP.Tag.CKA_LOCAL, "x-CKA_LOCAL"));
    register(new AttributeDefBool(KMIP.Tag.CKA_NEVER_EXTRACTABLE, "x-CKA_NEVER_EXTRACTABLE"));
    register(new AttributeDefBool(KMIP.Tag.CKA_ALWAYS_SENSITIVE, "x-CKA_ALWAYS_SENSITIVE"));
    register(new AttributeDefBool(KMIP.Tag.CKA_MODIFIABLE, "x-CKA_MODIFIABLE"));
    register(new AttributeDefBool(KMIP.Tag.CKA_ALWAYS_AUTHENTICATE, "x-CKA_ALWAYS_AUTHENTICATE"));
    register(new AttributeDefBytes(KMIP.Tag.CKA_EC_PARAMS, "x-CKA_EC_PARAMS"));
    register(new AttributeDefBytes(KMIP.Tag.CKA_EC_POINT, "x-CKA_EC_POINT"));
    register(new AttributeDefBool(KMIP.Tag.CKA_TRUSTED, "x-CKA_TRUSTED"));
    register(new AttributeDefBool(KMIP.Tag.CKA_WRAP_WITH_TRUSTED, "x-CKA_WRAP_WITH_TRUSTED"));

    register(new AttributeDefInt(KMIP.Tag.DYCKA_ECDSA_BIP_LEVEL, "x-DYCKA_ECDSA_BIP_LEVEL"));
    register(new AttributeDefInt(KMIP.Tag.DYCKA_ECDSA_BIP_CHILD_NUMBER, "x-DYCKA_ECDSA_BIP_CHILD_NUMBER"));
    register(new AttributeDefBytes(KMIP.Tag.DYCKA_ECDSA_BIP_PARENT_FINGERPRINT, "x-DYCKA_ECDSA_BIP_PARENT_FINGERPRINT"));
    register(new AttributeDefBytes(KMIP.Tag.DYCKA_ECDSA_BIP_CPAR, "x-DYCKA_ECDSA_BIP_CPAR"));
    register(new AttributeDefText(KMIP.Tag.DYCKA_ECDSA_BIP_PARENT_UID, "x-DYCKA_ECDSA_BIP_PARENT_UID"));
    register(new AttributeDefBool(KMIP.Tag.DYCKA_ECDSA_BIP_HARDENED, "x-DYCKA_ECDSA_BIP_HARDENED"));


    register(new AttributeDefInt(KMIP.Tag.DYCKA_EDDSA_BIP_LEVEL, "x-DYCKA_EDDSA_BIP_LEVEL"));
    register(new AttributeDefInt(KMIP.Tag.DYCKA_EDDSA_BIP_CHILD_NUMBER, "x-DYCKA_EDDSA_BIP_CHILD_NUMBER"));
    register(new AttributeDefBytes(KMIP.Tag.DYCKA_EDDSA_BIP_CPAR, "x-DYCKA_EDDSA_BIP_CPAR"));
    register(new AttributeDefText(KMIP.Tag.DYCKA_EDDSA_BIP_PARENT_UID, "x-DYCKA_EDDSA_BIP_PARENT_UID"));
    register(new AttributeDefBool(KMIP.Tag.DYCKA_EDDSA_BIP_HARDENED, "x-DYCKA_EDDSA_BIP_HARDENED"));

    register(new AttributeDefBool(KMIP.Tag.DyDYCKA_FIPS, "x-DYCKA_FIPS"));

    register(new AttributeDefBytes(KMIP.Tag.Key, "Key"));
    register(new AttributeDefBytes(KMIP.Tag.PublicKey, "Public Key"));
    register(new AttributeDefBytes(KMIP.Tag.PrivateKey, "Private Key"));

    register(new AttributeDefClass(KMIP.Tag.Link, "Link", Link.class));

    register(new AttributeDefInt(KMIP.Tag.DyChain, "x-DY_CHAIN"));
    register(new AttributeDefInt(KMIP.Tag.DyObjectGroup, "x-DY_OBJECT_GROUP"));
    register(new AttributeDefBool(KMIP.Tag.DyIsPrimaryObject, "x-DY_IS_PRIMARY_OBJECT"));
    register(new AttributeDefBool(KMIP.Tag.DyCreatedUsingDerive, "x-DY_CREATED-USING-DERIVE"));
    register(new AttributeDefInt(KMIP.Tag.DyPartitionId, "x-DY_PARTITION_ID"));
    register(new AttributeDefText(KMIP.Tag.DyPassword, "x-DY_PASSWORD"));
    register(new AttributeDefEnum(KMIP.Tag.DyAuthType, "x-DY_AUTH_TYPE"));
    register(new AttributeDefText(KMIP.Tag.DyDetails, "x-DY_DETAILS"));
    register(new AttributeDefInt(KMIP.Tag.DyActivationCode, "x-DY_ACTIVATION_CODE"));
    register(new AttributeDefText(KMIP.Tag.DyExpiration, "x-DY_ACTIVATION_CODE_EXPIRATION"));


    register(new AttributeDefBytes(KMIP.Tag.DyCertificateRequest, "x-DY-CERTIFICATE_REQUEST"));
    register(new AttributeDefText(KMIP.Tag.DyPfxPassword, "x-DY-PFX-PASSWORD"));
    register(new AttributeDefBytes(KMIP.Tag.DyPFX, "x-DY_PFX"));

    register(new AttributeDefText(KMIP.Tag.DyRole, "x-DY_ROLE"));
    register(new AttributeDefBytes(KMIP.Tag.DyBackup, "x-DY_BACKUP"));
    register(new AttributeDefBool(KMIP.Tag.DyIsSync, "x-DY_IS_SYNC"));
    register(new AttributeDefBool(KMIP.Tag.DyCKA_MPC_SIGN_PROTECTED, "x-DYCKA_MPC_SIGN_PROTECTED"));
    register(new AttributeDefBytes(KMIP.Tag.DyCKA_INTEGRITY_SIGNATURE, "x-DYCKA_INTEGRITY_SIGNATURE"));

    register(new AttributeDefInt(KMIP.Tag.DyClientRegistrainRetries, "x-DY_CLIENT_REGISTRATION_RETRIES"));
    register(new AttributeDefInt(KMIP.Tag.DyClientRegistrationTimeout, "x-DY_CLIENT_REGISTRATION_TIMEOUT"));
    register(new AttributeDefInt(KMIP.Tag.DyUserLoginRetries, "x-DY_USER_LOGIN_RETRIES"));
    register(new AttributeDefBool(KMIP.Tag.DyAllowOnlyCrypto, "x-DY_ALLOW_ONLY_CRYPTO"));
    register(new AttributeDefBool(KMIP.Tag.DyPasswordComplexity, "x-DY_PASSWORD_COMPLEXITY"));
    register(new AttributeDefInt(KMIP.Tag.DyPasswordLength, "x-DY_PASSWORD_LENGTH"));
    register(new AttributeDefBool(KMIP.Tag.DyCheckClientIp, "x-DY_CHECK_IP"));
    register(new AttributeDefBool(KMIP.Tag.DyAllowNat, "x-DY_ALLOW_NAT"));
    register(new AttributeDefText(KMIP.Tag.DyIpRange, "x-DY_IP_RANGE"));
    register(new AttributeDefInt64(KMIP.Tag.DyLastActivity, "x-DY_LAST_ACTIVITY"));
    register(new AttributeDefInt(KMIP.Tag.DySamlTimeout, "x-DY_SAML_TIMEOUT"));
    register(new AttributeDefBytes(KMIP.Tag.DySamlCertificate, "x-DY_SAML_CERTIFICATE"));
    register(new AttributeDefBool(KMIP.Tag.DyPartitionInheritance, "x-DY_PARTITION_INHERITANCE"));
    register(new AttributeDefInt(KMIP.Tag.DyJwtTimeout, "x-DY_JWT_EXPIRATION"));
    register(new AttributeDefInt(KMIP.Tag.DyJwtLimit, "x-DY_JWT_LIMIT"));
    register(new AttributeDefBool(KMIP.Tag.DyCertificatePropagation, "x-DY_CERTIFICATE_PROPAGATION"));
    register(new AttributeDefInt(KMIP.Tag.DyQuorumValue, "x-DY_QUORUM_VALUE"));
    register(new AttributeDefInt(KMIP.Tag.DyQuorumTimeout, "x-DY_QUORUM_TIMEOUT"));

    register(new AttributeDefBool(KMIP.Tag.DyEnforceUniqueName, "x-DY_ENFORCE_UNIQUE_NAME"));
    register(new AttributeDefBool(KMIP.Tag.DyEnforceUniqueDescription, "x-DY_ENFORCE_UNIQUE_DESCRIPTION"));

    register(new AttributeDefInt(KMIP.Tag.DyQuorumTimeout, "x-DY_QUORUM_TIMEOUT"));

    register(new AttributeDefBool(KMIP.Tag.DyCheckJwtOriginator, "x-DY_CHECK_JWT_ORIGINATOR"));

    register(new AttributeDefInt(KMIP.Tag.DyRefreshInterval, "x-DY_REFRESH_INTERVAL"));
    register(new AttributeDefInt(KMIP.Tag.DySyncInterval, "x-DY_SYNC_INTERVAL"));
    register(new AttributeDefBool(KMIP.Tag.DyCheckAllowServers, "x-DY_CHECK_ALLOW_SERVERS"));
    register(new AttributeDefBool(KMIP.Tag.DyAllowNoCertificate, "x-DY_ALLOW_NO_CERTIFICATE"));
    register(new AttributeDefInt(KMIP.Tag.DySeedInterval, "x-DY_SEED_INTERVAL"));
    register(new AttributeDefBytes(KMIP.Tag.DyLicense, "x-DY_LICENSE"));
    register(new AttributeDefBool(KMIP.Tag.DyFips, "x-DY_IS_FIPS"));
    register(new AttributeDefInt(KMIP.Tag.DyKeyRotationInterval, "x-DY_KEY_ROTATION_INTERVAL"));
    register(new AttributeDefInt64(KMIP.Tag.DyNextKeyRotationTime, "x-DY_NEXT_KEY_ROTATION_TIME"));
    register(new AttributeDefEnum(KMIP.Tag.DyPartitionFipsRequirements, "x-DY_FIPS_REQUIREMENTS"));

    register(new AttributeDefBool(KMIP.Tag.DyAllowDefaultClient, "x-DY_ALLOW_DEFAULT_CLIENT"));

    register(new AttributeDefBytes(KMIP.Tag.DyOfflineBackupKey, "x-DY_OFFLINE_BACKUP_KEY"));

    register(new AttributeDefBool(KMIP.Tag.DyGetAllCAs, "x-DY_GET_ALL_CAS"));
    register(new AttributeDefInt(KMIP.Tag.DyRootCaCertificateExpiration, "x-DY_ROOT_CA_CERTIFICATE_EXPIRATION"));
    register(new AttributeDefInt(KMIP.Tag.DyServerCertificateExpiration, "x-DY_SERVER_CERTIFICATE_EXPIRATION"));
    register(new AttributeDefInt(KMIP.Tag.DyClientCertificateExpiration, "x-DY_CLIENT_CERTIFICATE_EXPIRATION"));

    register(new AttributeDefInt(KMIP.Tag.DyRootCaCertificatePreExpiry, "x-DY_ROOT_CA_CERTIFICATE_PRE_EXPIRY"));
    register(new AttributeDefInt(KMIP.Tag.DyServerCertificatePreExpiry, "x-DY_SERVER_CERTIFICATE_PRE_EXPIRY"));
    register(new AttributeDefInt(KMIP.Tag.DyClientCertificatePreExpiry, "x-DY_CLIENT_CERTIFICATE_PRE_EXPIRY"));

    register(new AttributeDefInt(KMIP.Tag.DyActivationCodeLength, "x-DY_ACTIVATION_CODE_LENGTH"));

    register(new AttributeDefInt(KMIP.Tag.DyClientSecretExpiration, "x-DY_CLIENT_SECRET_EXPIRATION"));

    register(new AttributeDefInt(KMIP.Tag.DyExportable, "x-DY_EXPORTABLE"));
    register(new AttributeDefInt(KMIP.Tag.DyWrapable, "x-DY_WRAPABLE"));
    register(new AttributeDefInt(KMIP.Tag.DyWrapableWithTrusted, "x-DY_WRAPABLE_WITH_TRUSTED"));
    register(new AttributeDefInt(KMIP.Tag.DyNonExportable, "x-DY_NON_EXPORTABLE"));

    register(new AttributeDefText(KMIP.Tag.xID, "x-ID"));

    register(new AttributeDefBytes(KMIP.Tag.DYCKA_DIGEST_SHA_1, "x-DYCKA_DIGEST_SHA_1"));
    register(new AttributeDefBytes(KMIP.Tag.DYCKA_DIGEST_SHA256, "x-DYCKA_DIGEST_SHA256"));
    register(new AttributeDefBytes(KMIP.Tag.DYCKA_DIGEST_SHA384, "x-DYCKA_DIGEST_SHA384"));
    register(new AttributeDefBytes(KMIP.Tag.DYCKA_DIGEST_SHA512, "x-DYCKA_DIGEST_SHA512"));

    //register(new AttributeDefInt(KMIP.Tag.DYCKA_KEY_ROTATION_INTERVAL, "x-DYCKA_KEY_ROTATION_INTERVAL"));
    //register(new AttributeDefInt64(KMIP.Tag.DYCKA_NEXT_KEY_ROTATION_TIME, "x-DYCKA_NEXT_KEY_ROTATION_TIME"));


    register(new AttributeDefText(KMIP.Tag.DyLdapProviderUrl, "x-DY_LDAP_PROVIDER_URL"));
    register(new AttributeDefText(KMIP.Tag.DyLdapBindDn, "x-DY_LDAP_BIND_DN"));
    register(new AttributeDefText(KMIP.Tag.DyLdapBindDnPwd, "x-DY_LDAP_BIND_DN_PWD"));
    register(new AttributeDefBytes(KMIP.Tag.DyLdapCaCert, "x-DY_LDAP_CA_CERTIFICATE"));
    register(new AttributeDefText(KMIP.Tag.DyLdapSearchFilter, "x-DY_LDAP_SEARCH_FILTER"));
    register(new AttributeDefBool(KMIP.Tag.DyLdapDoNotVerifyCert, "x-DY_LDAP_DO_NOT_VERIFY_CERT"));
    register(new AttributeDefText(KMIP.Tag.DyKeyRotationExecTime, "x-DY_KEY_ROTATION_EXEC_TIME"));
    register(new AttributeDefText(KMIP.Tag.DyLdapSearchBase, "x-DY_LDAP_SEARCH_BASE"));


    register(new AttributeDefInt(KMIP.Tag.DyConnectTimeout, "x-DY_CONNECT_TIMEOUT"));
    register(new AttributeDefInt(KMIP.Tag.DyReadTimeout, "x-DY_READ_TIMEOUT"));
    register(new AttributeDefInt(KMIP.Tag.DyWriteTimeout, "x-DY_WRITE_TIMEOUT"));


    register(new AttributeDefInt(KMIP.Tag.DyKeepAliveIdle, "x-DY_KEEP_ALIVE_IDLE"));
    register(new AttributeDefInt(KMIP.Tag.DyKeepAliveInterval, "x-DY_KEEP_ALIVE_INTERVAL"));
    register(new AttributeDefInt(KMIP.Tag.DyKeepAliveCount, "x-DY_KEEP_ALIVE_COUNT"));

    register(new AttributeDefBytes(KMIP.Tag.DyJws, "x-DY_JWS"));

    register(new AttributeDefText(KMIP.Tag.DySecret, "x-DY_SECRET"));
    register(new AttributeDefInt(KMIP.Tag.DyGrantType, "x-DY_GRANT_TYPE"));
    register(new AttributeDefInt(KMIP.Tag.DyClientSecretExpiration, "x-DY_CLIENT_SECRET_EXPIRATION"));
    register(new AttributeDefText(KMIP.Tag.DYCKA_OBJECT_GROUPS, "x-DY_OBJECT_GROUPS"));

    register(new AttributeDefText(KMIP.Tag.DyDseVersion, "x-dse-version"));
    register(new AttributeDefText(KMIP.Tag.DyDseKeyNamespace, "x-key-namespace"));
  }


  public static AttributeDef getDef(int tag)
  {
    return tags.get(tag);
  }

  public static Attribute create(int tag)
  {
    AttributeDef def = tags.get(tag);
    if (def == null) return null;
    return def.create();
  }

  public static Attribute create(String name)
  {
    AttributeDef def = names.get(name);
    if (def == null) return null;
    return def.create();
  }

  public static int nameToTag(String name)
  {
    AttributeDef def = names.get(name);
    if (def == null) return 0;
    return def.tag;
  }

  public static String tagToName(int tag)
  {
    AttributeDef def = names.get(tag);
    if (def == null) return "";
    return def.name;
  }

  public static Attribute convert(KMIPConverter converter, Attribute object) throws KMIPConvertException
  {
    int begin = converter.convertBegin(KMIP.Tag.Attribute);

    if (converter.isWrite())
    {
      converter.convert(KMIP.Tag.AttributeName, object.getName());
    }
    else
    {
      String name = converter.convert(KMIP.Tag.AttributeName, (String) null);
      object = create(name);
      if (object == null) KMIPConverter.setError("Unknown attribute " + name);
      //object.tag = converter.getNextTag();
    }

    object.index = converter.convertOptional(KMIP.Tag.AttributeIndex, object.index);
    object.convert(converter);

    converter.convertEnd(begin);
    return object;
  }

  public abstract void convert(KMIPConverter converter) throws KMIPConvertException;
}
