package com.unbound.provider;

import com.unbound.common.Converter;
import com.unbound.common.Log;
import com.unbound.common.crypto.EC;
import com.unbound.common.crypto.SHA256;
import com.unbound.common.crypto.SystemProvider;
import com.unbound.provider.kmip.KMIP;
import com.unbound.provider.kmip.attribute.*;
import com.unbound.provider.kmip.object.ManagedObject;
import com.unbound.provider.kmip.object.PrivateKey;
import com.unbound.provider.kmip.request.*;
import com.unbound.provider.kmip.response.*;

import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.*;

public final class UBRSAPrivateKey extends UBPrivateKey implements RSAPrivateCrtKey
{
  UBRSAPrivateKey(Partition partition)
  {
    super(partition);
  }
  @Override

  int keyType()
  {
    return KMIP.CryptographicAlgorithm.RSA;
  }

  UBRSAPrivateKey(Partition partition, long uid, GetAttributesResponse getAttrResp, GetResponse get) throws InvalidKeySpecException
  {
    super(partition, uid, getAttrResp);
    Log log = Log.func("UBRSAPrivateKey").logHex("uid", uid).end(); try
    {
      pub = convertResponseToPublicKey(get);
    }
    catch (Exception e) { log.failed(e); throw e; } finally { log.leave(); }
  }

  UBRSAPrivateKey(Partition partition, String alias, RSAPrivateCrtKey key) throws InvalidKeySpecException, IOException
  {
    super(partition);
    register(alias, KMIP.CryptographicAlgorithm.RSA,
      KMIP.CryptographicUsageMask.Sign | KMIP.CryptographicUsageMask.Decrypt,
      prepareManagedObject(key));
  }

  UBRSAPrivateKey(Partition partition, KeySpec keySpec) throws InvalidKeySpecException, IOException
  {
    super(partition);
    register(null, KMIP.CryptographicAlgorithm.RSA,
      KMIP.CryptographicUsageMask.Sign | KMIP.CryptographicUsageMask.Decrypt,
      prepareManagedObject(keySpec));
  }

  @Override
  PublicKey convertResponseToPublicKey(GetResponse get) throws InvalidKeySpecException
  {
    com.unbound.provider.kmip.object.PrivateKey kmipManagedObject = (com.unbound.provider.kmip.object.PrivateKey)get.object;
    X509EncodedKeySpec spec = new X509EncodedKeySpec(kmipManagedObject.keyBlock.buf);
    KeyFactory kf = SystemProvider.KeyFactory.getInstance("RSA");
    return kf.generatePublic(spec);
  }

  @Override
  TemplateAttribute prepareGenerateTemplate(int bitSize, EC.Curve curve)
  {
    int usage =
      KMIP.CryptographicUsageMask.Sign |
      KMIP.CryptographicUsageMask.Decrypt;
    TemplateAttribute template = new TemplateAttribute();
    template.attrs.add(new EnumAttribute(KMIP.Tag.CryptographicAlgorithm, KMIP.CryptographicAlgorithm.RSA));
    template.attrs.add(new IntAttribute(KMIP.Tag.CryptographicLength, bitSize));
    template.attrs.add(new IntAttribute(KMIP.Tag.CryptographicUsageMask, usage));
    return template;
  }

  private ManagedObject prepareManagedObjectCrt(RSAPrivateCrtKeySpec keySpec)
  {
    PrivateKey mo = new PrivateKey();
    mo.keyBlock.formatType = KMIP.KeyFormatType.TransparentRSAPrivateKey;
    mo.keyBlock.N = keySpec.getModulus();
    mo.keyBlock.E = keySpec.getPublicExponent();
    mo.keyBlock.D = keySpec.getPrivateExponent();
    mo.keyBlock.P = keySpec.getPrimeP();
    mo.keyBlock.Q = keySpec.getPrimeP();
    mo.keyBlock.DP = keySpec.getPrimeExponentP();
    mo.keyBlock.DQ = keySpec.getPrimeExponentQ();
    mo.keyBlock.QINV = keySpec.getCrtCoefficient();
    return mo;
  }

  private ManagedObject prepareManagedObjectPkcs8(PKCS8EncodedKeySpec keySpec)
  {
    PrivateKey mo = new PrivateKey();
    mo.keyBlock.formatType = KMIP.KeyFormatType.PKCS_8;
    mo.keyBlock.buf = keySpec.getEncoded();
    return mo;
  }

  private ManagedObject prepareManagedObject(KeySpec keySpec)
  {
    if (keySpec instanceof RSAPrivateCrtKeySpec) return prepareManagedObjectCrt((RSAPrivateCrtKeySpec)keySpec);
    if (keySpec instanceof PKCS8EncodedKeySpec) return prepareManagedObjectPkcs8((PKCS8EncodedKeySpec)keySpec);
    return null;
  }

  private ManagedObject prepareManagedObject(RSAPrivateCrtKey key)
  {
    PrivateKey mo = new PrivateKey();
    mo.keyBlock.formatType = KMIP.KeyFormatType.TransparentRSAPrivateKey;
    mo.keyBlock.N = key.getModulus();
    mo.keyBlock.E = key.getPublicExponent();
    mo.keyBlock.D = key.getPrivateExponent();
    mo.keyBlock.P = key.getPrimeP();
    mo.keyBlock.Q = key.getPrimeP();
    mo.keyBlock.DP = key.getPrimeExponentP();
    mo.keyBlock.DQ = key.getPrimeExponentQ();
    mo.keyBlock.QINV = key.getCrtCoefficient();
    return mo;
  }

  RSAPublicKey rsaPublicKey()
  {
    return (RSAPublicKey)pub;
  }

  int getBitSize()
  {
    return rsaPublicKey().getModulus().bitLength();
  }

  static long getKeyUid(RSAPublicKey pub)
  {
    byte[] hash = SHA256.hash(pub.getModulus().toByteArray());
    return Converter.getBE8(hash, 0);
  }

  byte[] decrypt(byte[] in, int kmipPadding, int kmipHashAlg, int kmipMgfAlg, byte[] label) throws IOException
  {
    DecryptRequest req = new DecryptRequest();
    req.uid = uidToStr(uid);
    req.params = new CryptoParams();
    req.params.cryptoAlg = KMIP.CryptographicAlgorithm.RSA;
    req.params.padding = kmipPadding;
    req.data = in;
    if (kmipPadding==KMIP.PaddingMethod.OAEP)
    {
      req.params.hashingAlg = kmipHashAlg;
      req.ext = new MessageExt();
      req.ext.mgf = kmipMgfAlg;
      if (label!=null && label.length!=0) req.ext.auth.add(label);
    }

    DecryptResponse resp = (DecryptResponse) partition.transmit(req);
    return resp.data;
  }

  // ------------------------------ interface -----------------

  @Override
  public BigInteger getPublicExponent()
  {
    return rsaPublicKey().getPublicExponent();
  }

  @Override
  public BigInteger getPrimeP()
  {
    return null;
  }

  @Override
  public BigInteger getPrimeQ()
  {
    return null;
  }

  @Override
  public BigInteger getPrimeExponentP()
  {
    return null;
  }

  @Override
  public BigInteger getPrimeExponentQ()
  {
    return null;
  }

  @Override
  public BigInteger getCrtCoefficient()
  {
    return null;
  }

  @Override
  public BigInteger getPrivateExponent()
  {
    return null;
  }

  @Override
  public String getAlgorithm()
  {
    return "RSA";
  }

  @Override
  public String getFormat()
  {
    return "N/A";
  }

  @Override
  public byte[] getEncoded()
  {
    return null;
  }

  @Override
  public BigInteger getModulus()
  {
    return rsaPublicKey().getModulus();
  }
}
