package com.unbound.provider;

import com.unbound.common.Log;
import com.unbound.common.crypto.EC;
import com.unbound.provider.kmip.KMIP;
import com.unbound.provider.kmip.attribute.*;
import com.unbound.provider.kmip.object.ManagedObject;
import com.unbound.provider.kmip.request.*;
import com.unbound.provider.kmip.response.*;

import java.io.IOException;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.InvalidKeySpecException;

abstract class UBPrivateKey extends UBObject implements PrivateKey
{
  PublicKey pub = null;

  UBPrivateKey(Partition partition)
  {
    super(partition);
  }

  @Override
  int objectType()
  {
    return KMIP.ObjectType.PrivateKey;
  }

  abstract int keyType();

  UBPrivateKey(Partition partition, long uid, GetAttributesResponse getAttrResp)
  {
    super(partition, uid, getAttrResp);
  }

  static UBPrivateKey newPrivateKey(Partition partition, long uid, GetAttributesResponse getAttrResp, GetResponse getResp) throws InvalidKeySpecException
  {
    int keyType = ((EnumAttribute)getAttrResp.attrs.get(1)).value;
    switch (keyType)
    {
      case KMIP.CryptographicAlgorithm.EC:    return new UBECPrivateKey(partition, uid, getAttrResp, getResp);
      case KMIP.CryptographicAlgorithm.RSA:   return new UBRSAPrivateKey(partition, uid, getAttrResp, getResp);
      default:                                throw new ProviderException("Unsupported key type");
    }
  }

  static UBPrivateKey newPrivateKey(Partition partition, String alias, Key key) throws InvalidKeySpecException, IOException
  {
    if (key instanceof RSAPrivateCrtKey) return new UBRSAPrivateKey(partition, alias, (RSAPrivateCrtKey)key);
    if (key instanceof ECPrivateKey) return new UBECPrivateKey(partition, alias, (ECPrivateKey)key);
    throw new ProviderException("Unsupported key type");
  }

  abstract PublicKey convertResponseToPublicKey(GetResponse resp) throws InvalidKeySpecException;

  abstract TemplateAttribute prepareGenerateTemplate(int bitSize, EC.Curve curve);

  void generate(int bitSize, EC.Curve curve) throws InvalidKeySpecException, IOException
  {
    Log log = Log.func("UBPrivateKey.generate").log("bitSize", bitSize).end(); try
    {
      RequestMessage reqMsg = new RequestMessage();

      CreateKeyPairRequest reqCreateKeyPair = new CreateKeyPairRequest();
      reqMsg.batch.add(reqCreateKeyPair);
      reqCreateKeyPair.prv = prepareGenerateTemplate(bitSize, curve);

      ActivateRequest reqActivate = new ActivateRequest();
      reqMsg.batch.add(reqActivate);

      GetAttributesRequest reqGetAttr = new GetAttributesRequest();
      reqMsg.batch.add(reqGetAttr);
      reqGetAttr.names.add("Name");
      reqGetAttr.names.add("Initial Date");

      GetRequest reqGet = new GetRequest();
      reqMsg.batch.add(reqGet);
      reqGet.formatType = KMIP.KeyFormatType.X_509;

      ResponseMessage respMsg = partition.transmit(reqMsg);

      CreateKeyPairResponse respCreateKeyPair = (CreateKeyPairResponse)respMsg.batch.get(0);
      GetAttributesResponse respGetAttr = (GetAttributesResponse)respMsg.batch.get(2);
      GetResponse respGet = (GetResponse)respMsg.batch.get(3);

      uid = strToUid(respCreateKeyPair.prvUID);
      name = ((Name)respGetAttr.attrs.get(0)).value;
      initialDate = ((DateAttribute)respGetAttr.attrs.get(1)).value;
      pub = convertResponseToPublicKey(respGet);
    }
    catch (Exception e) { log.failed(e); throw e; } finally { log.leavePrint().logHex("UID", uid).end(); }
  }

  void register(String alias, int kmipAlg, int kmipUsage, ManagedObject managedObject) throws InvalidKeySpecException, IOException
  {
    RequestMessage reqMsg = new RequestMessage();

    RegisterRequest reqRegister = new RegisterRequest();
    reqMsg.batch.add(reqRegister);
    reqRegister.objectType = KMIP.ObjectType.PrivateKey;
    reqRegister.template = new TemplateAttribute();
    reqRegister.template.attrs.add(new EnumAttribute(KMIP.Tag.CryptographicAlgorithm, kmipAlg));
    reqRegister.template.attrs.add(new IntAttribute(KMIP.Tag.CryptographicUsageMask, kmipUsage));
    if (alias!=null) reqRegister.template.attrs.add(new Name(alias));
    reqRegister.object = managedObject;

    ActivateRequest reqActivate = new ActivateRequest();
    reqMsg.batch.add(reqActivate);

    GetAttributesRequest reqGetAttr = new GetAttributesRequest();
    reqMsg.batch.add(reqGetAttr);
    reqGetAttr.names.add("Name");
    reqGetAttr.names.add("Initial Date");

    GetRequest reqGet = new GetRequest();
    reqMsg.batch.add(reqGet);
    reqGet.formatType = KMIP.KeyFormatType.X_509;

    ResponseMessage respMsg = partition.transmit(reqMsg);

    RegisterResponse respRegister = (RegisterResponse)respMsg.batch.get(0);
    GetAttributesResponse respGetAttr = (GetAttributesResponse)respMsg.batch.get(2);
    GetResponse respGet = (GetResponse)respMsg.batch.get(3);

    uid = strToUid(respRegister.uid);
    name = alias==null ? ((Name)respGetAttr.attrs.get(0)).value : alias;
    initialDate = ((DateAttribute)respGetAttr.attrs.get(1)).value;
    pub = convertResponseToPublicKey(respGet);
  }

  static UBPrivateKey locate(Partition partition, String alias) throws InvalidKeySpecException, CertificateException, IOException
  {
    long uid = 0;
    Log log = Log.func("UBPrivateKey.locate").log("alias", alias).end(); try
    {
      uid = partition.locate(KMIP.ObjectType.PrivateKey, 0, alias);
      if (uid==0) return null;
      return (UBPrivateKey)UBObject.read(partition, uid);
    } catch (Exception e) { log.failed(e); throw e; } finally { log.leavePrint().logHex("uid", uid).end(); }
  }

  byte[] sign(byte[] in, int kmipAlg, int kmipSigAlg) throws IOException
  {
    Log log = Log.func("UBPrivateKey.sign").log("kmipAlg", kmipAlg).log("kmipSigAlg", kmipSigAlg).log("in.length", in.length).end(); try
    {
      SignRequest req = new SignRequest();
      req.uid = uidToStr(uid);
      req.data = in;
      req.params = new CryptoParams();
      req.params.cryptoAlg = kmipAlg;
      if (kmipAlg==KMIP.CryptographicAlgorithm.RSA) req.params.padding = KMIP.PaddingMethod.PKCS1_V1_5;
      if (kmipSigAlg!=0) req.params.signingAlg = kmipSigAlg;

      SignResponse resp = (SignResponse) partition.transmit(req);
      return resp.data;
    } catch (Exception e) { log.failed(e); throw e; } finally { log.leave(); }
  }

}
