package com.unbound.provider;

import com.unbound.common.HEX;
import com.unbound.provider.kmip.KMIP;
import com.unbound.provider.kmip.attribute.DateAttribute;
import com.unbound.provider.kmip.attribute.EnumAttribute;
import com.unbound.provider.kmip.attribute.Name;
import com.unbound.provider.kmip.request.*;
import com.unbound.provider.kmip.response.GetAttributesResponse;
import com.unbound.provider.kmip.response.GetResponse;
import com.unbound.provider.kmip.response.ResponseMessage;

import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.ProviderException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;

import static com.unbound.common.Converter.setBE8;

abstract class UBObject
{
  Partition partition;
  long uid;
  String name;
  long initialDate;

  static long strToUid(String str)
  {
    if (str.length() == 16) str = "0x00" + str;
    if (str.length() == 20 && str.charAt(0) == '0' && str.charAt(1) == 'x' && str.charAt(2) == '0' && str.charAt(3) == '0')
    {
      long uid = 0;
      for (int i = 4; i < 20; i += 2)
      {
        int hi = HEX.from(str.charAt(i));
        int lo = HEX.from(str.charAt(i + 1));
        if (lo < 0 || hi < 0) return 0;
        uid = (uid << 8) | (hi << 4) | lo;
      }
      return uid;
    }

    return 0;
  }

  static String uidToStr(long uid)
  {
    byte[] cka_id = new byte[9];
    cka_id[0] = 0;
    setBE8(cka_id, 1, uid);

    char[] out = new char[20];
    out[0] = '0';
    out[1] = 'x';

    for (int i = 0; i < cka_id.length; i++)
    {
      out[2 + i * 2 + 0] = HEX.chars[(cka_id[i] >> 4) & 0x0f];
      out[2 + i * 2 + 1] = HEX.chars[cka_id[i] & 0x0f];
    }

    return new String(out);
  }


  UBObject(Partition partition)
  {
    this.partition = partition;
  }

  UBObject(Partition partition, long uid, GetAttributesResponse getAttrResp)
  {
    this.partition = partition;
    this.uid = uid;
    name = ((Name)getAttrResp.attrs.get(2)).value;
    initialDate = ((DateAttribute)getAttrResp.attrs.get(3)).value;
  }

  abstract int objectType();

  static UBObject[] read(Partition partition, long[] uids) throws CertificateException, NoSuchAlgorithmException, InvalidKeySpecException, IOException
  {
    if (uids.length==0) return new UBObject[0];

    ResponseMessage respMsg = partition.read(uids);
    UBObject[] list = new UBObject[uids.length];

    for (int i=0; i<uids.length; i++)
    {
      GetAttributesResponse getAttrResp = (GetAttributesResponse)respMsg.batch.get(i*2);
      GetResponse getResp = (GetResponse)respMsg.batch.get(i*2+1);
      list[i] = newObject(partition, uids[i], getAttrResp, getResp);
    }

    return list;
  }

  static UBObject read(Partition partition, long uid) throws CertificateException, InvalidKeySpecException, IOException
  {
    ResponseMessage respMsg = partition.read(uid);
    GetAttributesResponse getAttrResp = (GetAttributesResponse)respMsg.batch.get(0);
    GetResponse getResp = (GetResponse)respMsg.batch.get(1);
    return newObject(partition, uid, getAttrResp, getResp);
  }

  private static UBObject newObject(Partition partition, long uid, GetAttributesResponse getAttrResp, GetResponse getResp) throws CertificateException, InvalidKeySpecException
  {
    int objectType = ((EnumAttribute)getAttrResp.attrs.get(0)).value;

    switch (objectType)
    {
      case KMIP.ObjectType.PrivateKey:    return UBPrivateKey.newPrivateKey(partition, uid, getAttrResp, getResp);
      case KMIP.ObjectType.Certificate:   return new UBCertificate(partition, uid, getAttrResp, getResp);
      default:                            throw new ProviderException("Unsupported object type");
    }
  }

  static LocateRequest locateRequest(int objectType, int keyType, String alias)
  {
    LocateRequest req = new LocateRequest();
    req.maxItems = 1;
    req.attrs.add(new EnumAttribute(KMIP.Tag.State, KMIP.State.Active));
    req.attrs.add(new EnumAttribute(KMIP.Tag.ObjectType, objectType));
    if (keyType != 0) req.attrs.add(new EnumAttribute(KMIP.Tag.CryptographicAlgorithm, keyType));
    if (alias!=null) req.attrs.add(new Name(alias));
    return req;
  }

  static void delete(Partition partition, UBObject obj1) throws IOException
  { delete(partition, obj1, null, null); }
  static void delete(Partition partition, UBObject obj1, UBObject obj2) throws IOException
  { delete(partition, obj1, obj2, null); }
  static void delete(Partition partition, UBObject obj1, UBObject obj2, UBObject obj3) throws IOException
  {
    RequestMessage reqMsg = new RequestMessage();

    RevokeRequest r = new RevokeRequest(); r.uid = uidToStr(obj1.uid);
    DestroyRequest d = new DestroyRequest();
    reqMsg.batch.add(r); reqMsg.batch.add(d);

    if (obj2!=null)
    {
      r = new RevokeRequest(); r.uid = uidToStr(obj2.uid);
      d = new DestroyRequest();
      reqMsg.batch.add(r); reqMsg.batch.add(d);
    }

    if (obj3!=null)
    {
      r = new RevokeRequest(); r.uid = uidToStr(obj3.uid);
      d = new DestroyRequest();
      reqMsg.batch.add(r); reqMsg.batch.add(d);
    }


    partition.transmit(reqMsg);
  }

  void setName(String name) throws IOException
  {
    if (this.name.equals(name)) return;

    ModifyAttributeRequest req = new ModifyAttributeRequest();
    req.uid = uidToStr(uid);
    req.attribute = new Name(name);
    partition.transmit(req);
    this.name = name;
  }
}
