package com.unbound.provider;

import com.unbound.common.Log;
import com.unbound.provider.kmip.KMIP;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.*;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.time.Clock;
import java.util.*;

public final class UBKeyStore extends KeyStoreSpi
{
  private static final int CACHE_TIMEOUT = 30000; // milliseconds
  private static final Clock clock = Clock.systemUTC();

  static final class Entry
  {
    UBPrivateKey key = null;
    UBCertificate cert = null;
    Certificate[] chain;

    Entry(UBPrivateKey key, UBCertificate cert, Certificate[] chain) throws InvalidKeySpecException, CertificateException, IOException
    {
      this.key = key;
      this.cert = cert;
      if (chain==null) chain = cert.getChain();
      this.chain = chain;
    }

    Entry(UBCertificate cert)
    {
      this.cert = cert;
      this.chain = null;
    }

    String getName()
    {
      return isTrustedCert() ? cert.name : key.name;
    }

    boolean isTrustedCert()
    {
      return key==null;
    }
  }

  private Partition partition;
  private Map<String, Entry> cache = new HashMap<>();
  private long lastCacheClock = 0;

  UBKeyStore(Partition partition)
  {
    this.partition = partition;
  }

  private Entry findEntry(String alias) throws InvalidKeySpecException, CertificateException, IOException
  {
    Entry entry = null;
    Log log = Log.func("UBKeyStore.findEntry").log("alias", alias).end(); try
    {
      synchronized (this)
      {
        entry = cache.get(alias.toUpperCase());
      }
      if (entry!=null) return entry;

      UBPrivateKey key = UBPrivateKey.locate(partition, alias);
      if (key==null)
      {
        UBCertificate cert = UBCertificate.locate(partition, alias);
        if (cert==null) return null;
        entry = new Entry(cert);
      }
      else
      {
        UBCertificate cert = UBCertificate.locateByKeyUid(partition, key.uid);
        if (cert==null) return null;
        entry = new Entry(key, cert, null);
      }

      synchronized (this)
      {
        cache.put(entry.getName().toUpperCase(), entry);
      }
      return entry;
    }
    catch (Exception e) { log.failed(e); throw e; }
    finally
    {
      log = log.leavePrint();
      if (entry!=null) log.log("entry", entry.getName());
      log.end();
    }
  }

  // -------------------- interface ------------------------

  @Override
  public Key engineGetKey(String alias, char[] password) throws NoSuchAlgorithmException, UnrecoverableKeyException
  {
    Entry entry;
    try
    {
      if (password!=null && password.length!=0) partition.login(new String(password));
      entry = findEntry(alias);
    }
    catch (CertificateException | InvalidKeySpecException | IOException e)
    {
      throw new ProviderException(e);
    }
    if (entry==null) return null;
    return entry.key;
  }

  @Override
  public Certificate[] engineGetCertificateChain(String alias)
  {
    Entry entry;
    try { entry = findEntry(alias); }
    catch (Exception e) { throw new ProviderException(e); }
    if (entry==null) return null;
    return entry.chain;
  }

  @Override
  public Certificate engineGetCertificate(String alias)
  {
    Entry entry;
    try { entry = findEntry(alias); }
    catch (Exception e) { throw new ProviderException(e); }
    if (entry==null) return null;
    return entry.cert.x509;
  }

  @Override
  public Date engineGetCreationDate(String alias)
  {
    Entry entry;
    try { entry = findEntry(alias); }
    catch (Exception e) { throw new ProviderException(e); }
    if (entry==null) return null;
    long initialDate = entry.isTrustedCert() ? entry.cert.initialDate : entry.key.initialDate;
    if (initialDate<0) return null;
    return new Date(initialDate * 1000);
  }

  @Override
  public void engineSetKeyEntry(String alias, Key key, char[] password, Certificate[] chain) throws KeyStoreException
  {
    try
    {
      if (password!=null && password.length!=0) partition.login(new String(password));

      UBPrivateKey privateKey = (key instanceof UBPrivateKey) ? (UBPrivateKey)key : null;

      Entry entry = findEntry(alias);
      if (entry!=null)
      {
        if (entry.isTrustedCert()) throw new KeyStoreException("Trusted certificate entry present");
        if (privateKey==null) engineDeleteEntry(alias);
      }

      if (privateKey==null) privateKey = UBPrivateKey.newPrivateKey(partition, alias, key);
      else privateKey.setName(alias);

      X509Certificate x509 = (X509Certificate)chain[0];
      UBCertificate cert = UBCertificate.locate(partition, x509);
      if (cert==null) cert = new UBCertificate(partition, alias, x509);
      else cert.setName(alias);

      for (int i=1; i<chain.length; i++)
      {
        x509 = (X509Certificate)chain[1];
        UBCertificate chainCert = UBCertificate.locate(partition, x509);
        if (chainCert==null) new UBCertificate(partition, alias, x509);
      }

      entry = new Entry(privateKey, cert, chain);
      synchronized (this)
      {
        cache.put(alias, entry);
      }
    }
    catch (IOException | CertificateException | InvalidKeySpecException e)
    {
      throw new KeyStoreException(e);
    }
  }

  @Override
  public void engineSetKeyEntry(String alias, byte[] key, Certificate[] chain) throws KeyStoreException
  {
    throw new KeyStoreException("Not supported");
  }

  @Override
  public void engineSetCertificateEntry(String alias, Certificate cert) throws KeyStoreException
  {
    try
    {
      Entry entry = findEntry(alias);
      if (entry!=null)
      {
        if (!entry.isTrustedCert()) throw new KeyStoreException("Private key entry present");
        engineDeleteEntry(alias);
      }
      UBCertificate c = new UBCertificate(partition, alias, (X509Certificate)cert);
      entry = new Entry(c);
      synchronized (this)
      {
        cache.put(alias, entry);
      }
    }
    catch (IOException | CertificateException | InvalidKeySpecException e)
    {
      throw new KeyStoreException(e);
    }
  }

  @Override
  public void engineDeleteEntry(String alias) throws KeyStoreException
  {
    Entry entry;
    try
    {
      entry = findEntry(alias);
      if (entry==null) return;
      UBObject.delete(partition, entry.cert, entry.key);
      synchronized (this) { cache.remove(alias); }
    }
    catch (IOException | CertificateException | InvalidKeySpecException e)
    {
      throw new KeyStoreException(e);
    }
  }

  @Override
  public Enumeration<String> engineAliases()
  {
    synchronized (this)
    {
      long now = clock.millis();
      if (now < lastCacheClock + CACHE_TIMEOUT) return Collections.enumeration(cache.keySet());
    }

    try
    {
      long[] rsaIds = partition.locate(KMIP.ObjectType.PrivateKey, KMIP.CryptographicAlgorithm.RSA);
      long[] eccIds = partition.locate(KMIP.ObjectType.PrivateKey, KMIP.CryptographicAlgorithm.EC);
      long[] keyIds = new long[rsaIds.length + eccIds.length];
      System.arraycopy(rsaIds, 0, keyIds, 0, rsaIds.length);
      System.arraycopy(eccIds, 0, keyIds, rsaIds.length, eccIds.length);
      long[] certIds = partition.locate(KMIP.ObjectType.Certificate, 0);

      UBObject[] keys = UBObject.read(partition, keyIds);
      UBObject[] certs = UBObject.read(partition, certIds);

      Map<String, Entry> cache = new HashMap<>();

      for (UBObject object : keys)
      {
        UBPrivateKey key = (UBPrivateKey)object;
        UBCertificate found = null;
        for (int i=0; i<certs.length; i++)
        {
          UBCertificate cert = (UBCertificate)certs[i];
          if (cert==null) continue;
          if (cert.match(key.pub)) { found = cert; certs[i]=null; break; }
        }

        if (found!=null)
        {
          Entry entry = new Entry(key, found, found.getChain(certs));
          cache.put(key.name.toUpperCase(), entry);
        }
      }

      synchronized (this)
      {
        this.cache = cache;
        lastCacheClock = clock.millis();
      }
      return Collections.enumeration(cache.keySet());
    }
    catch (IOException | CertificateException | NoSuchAlgorithmException | InvalidKeySpecException e)
    {
      throw new ProviderException(e);
    }
  }

  @Override
  public boolean engineContainsAlias(String alias)
  {
    Entry entry = null;
    try { entry = findEntry(alias); }
    catch (Exception e) { }
    return entry!=null;
  }

  @Override
  public int engineSize()
  {
    return 0;
  }

  @Override
  public boolean engineIsKeyEntry(String alias)
  {
    Entry entry = null;
    try { entry = findEntry(alias); }
    catch (Exception e) {  }
    return entry!=null && !entry.isTrustedCert();
  }

  @Override
  public boolean engineIsCertificateEntry(String alias)
  {
    Entry entry = null;
    try { entry = findEntry(alias); }
    catch (Exception e) { }
    return entry!=null && entry.isTrustedCert();
  }

  @Override
  public String engineGetCertificateAlias(Certificate cert)
  {
    if (!(cert instanceof X509Certificate)) return null;
    X509Certificate x509 = (X509Certificate)cert;

    synchronized (this)
    {
      for (Entry entry: cache.values())
      {
        if (entry.cert.x509.equals(cert)) return entry.getName();
      }
    }

    try
    {
      UBCertificate certificate = UBCertificate.locate(partition, x509);
      if (certificate==null) return null;

      long keyUid = UBCertificate.getKeyUid(x509);
      UBPrivateKey key = null;
      try { key = (UBPrivateKey)UBObject.read(partition, keyUid); }
      catch (Exception e) {}
      Entry entry = (key==null) ? new Entry(certificate) : new Entry(key, certificate, null);

      synchronized (this)
      {
        cache.put(entry.getName().toUpperCase(), entry);
      }
      return entry.getName();
    }
    catch (Exception e) { }
    return null;
  }

  @Override
  public void engineStore(OutputStream stream, char[] password) throws IOException, NoSuchAlgorithmException, CertificateException
  {
    // nothing to do
  }

  @Override
  public void engineLoad(InputStream stream, char[] password) throws IOException, NoSuchAlgorithmException, CertificateException
  {
    if (password!=null && password.length!=0) partition.login(new String(password));
  }
}
