package com.unbound.provider;

import com.unbound.common.Converter;
import com.unbound.common.Log;
import com.unbound.common.crypto.EC;
import com.unbound.common.crypto.SHA256;
import com.unbound.common.crypto.SystemProvider;
import com.unbound.provider.kmip.KMIP;
import com.unbound.provider.kmip.attribute.*;
import com.unbound.provider.kmip.object.ManagedObject;
import com.unbound.provider.kmip.object.PrivateKey;
import com.unbound.provider.kmip.response.GetAttributesResponse;
import com.unbound.provider.kmip.response.GetResponse;

import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.*;

public final class UBECPrivateKey extends UBPrivateKey implements java.security.interfaces.ECPrivateKey
{
  UBECPrivateKey(Partition partition)
  {
    super(partition);
  }

  @Override
  int keyType()
  {
    return KMIP.CryptographicAlgorithm.EC;
  }

  UBECPrivateKey(Partition partition, String alias, ECPrivateKey key) throws InvalidKeySpecException, IOException
  {
    super(partition);
    Log log = Log.func("UBECPrivateKey").log("alias", alias).end(); try
    {
      register(alias, KMIP.CryptographicAlgorithm.EC, KMIP.CryptographicUsageMask.Sign, prepareManagedObject(key));
    }
    catch (Exception e) { log.failed(e); throw e; } finally { log.leave(); }
  }

  UBECPrivateKey(Partition partition, KeySpec keySpec) throws InvalidKeySpecException, IOException
  {
    super(partition);
    register(null, KMIP.CryptographicAlgorithm.EC, KMIP.CryptographicUsageMask.Sign, prepareManagedObject(keySpec));
  }

  UBECPrivateKey(Partition partition, long uid, GetAttributesResponse getAttrResp, GetResponse get) throws InvalidKeySpecException
  {
    super(partition, uid, getAttrResp);
    Log log = Log.func("UBECPrivateKey").logHex("uid", uid).end(); try
    {
      pub = convertResponseToPublicKey(get);
    }
    catch (Exception e) { log.failed(e); throw e; } finally { log.leave(); }
  }

  @Override
  PublicKey convertResponseToPublicKey(GetResponse get) throws InvalidKeySpecException
  {
    PrivateKey kmipManagedObject = (PrivateKey)get.object;
    X509EncodedKeySpec spec = new X509EncodedKeySpec(kmipManagedObject.keyBlock.buf);
    KeyFactory kf = SystemProvider.KeyFactory.getInstance("EC");
    return kf.generatePublic(spec);
  }

  public ECPublicKey ecPublicKey()
  {
    return (ECPublicKey)pub;
  }

  EC.Curve getCurve()
  {
    return EC.getCurve(ecPublicKey());
  }

  int size()
  {
    return getCurve().size;
  }

  @Override
  TemplateAttribute prepareGenerateTemplate(int bitSize, EC.Curve curve)
  {
    int usage = KMIP.CryptographicUsageMask.Sign;
    TemplateAttribute template = new TemplateAttribute();
    template.attrs.add(new EnumAttribute(KMIP.Tag.CryptographicAlgorithm, KMIP.CryptographicAlgorithm.EC));
    template.attrs.add(new EnumAttribute(KMIP.Tag.RecommendedCurve, curve.kmipCode));
    template.attrs.add(new IntAttribute(KMIP.Tag.CryptographicUsageMask, usage));
    return template;
  }

  private ManagedObject prepareManagedObjectRaw(ECPrivateKeySpec keySpec) throws InvalidKeySpecException
  {
    EC.Curve curve = EC.getCurve(keySpec.getParams());
    if (curve==null) throw new InvalidKeySpecException("Curve not supported");

    PrivateKey mo = new PrivateKey();
    mo.keyBlock.formatType = KMIP.KeyFormatType.TransparentECPrivateKey;
    mo.keyBlock.D = keySpec.getS();
    mo.keyBlock.curveType = curve.kmipCode;
    return mo;
  }

  private ManagedObject prepareManagedObjectPkcs8(PKCS8EncodedKeySpec keySpec)
  {
    PrivateKey mo = new PrivateKey();
    mo.keyBlock.formatType = KMIP.KeyFormatType.PKCS_8;
    mo.keyBlock.buf = keySpec.getEncoded();
    return mo;
  }

  private ManagedObject prepareManagedObject(KeySpec keySpec) throws InvalidKeySpecException
  {
    if (keySpec instanceof ECPrivateKeySpec) return prepareManagedObjectRaw((ECPrivateKeySpec)keySpec);
    if (keySpec instanceof PKCS8EncodedKeySpec) return prepareManagedObjectPkcs8((PKCS8EncodedKeySpec)keySpec);
    return null;
  }

  private ManagedObject prepareManagedObject(ECPrivateKey key) throws InvalidKeySpecException
  {
    EC.Curve curve = EC.getCurve(key.getParams());
    if (curve==null) throw new InvalidKeySpecException("Curve not supported");

    PrivateKey mo = new PrivateKey();
    mo.keyBlock.formatType = KMIP.KeyFormatType.TransparentECPrivateKey;
    mo.keyBlock.D = key.getS();
    mo.keyBlock.curveType = curve.kmipCode;
    return mo;
  }

  static long getKeyUid(ECPublicKey pub)
  {
    EC.Curve curve = EC.getCurve(pub);
    byte[] hash = SHA256.hash(curve.toDer(pub.getW()));
    return Converter.getBE8(hash, 0);
  }

// ------------------------- interface -------------------
  @Override
  public BigInteger getS()
  {
    return null;
  }

  @Override
  public String getAlgorithm() { return "EC"; }

  @Override
  public String getFormat() { return "N/A"; }

  @Override
  public byte[] getEncoded()
  {
    return null;
  }

  @Override
  public ECParameterSpec getParams()
  {
    return ecPublicKey().getParams();
  }
}
