package com.unbound.provider;

import com.unbound.common.Config;
import com.unbound.common.Log;

import java.io.IOException;
import java.net.MalformedURLException;
import java.security.*;
import java.security.cert.CertificateException;
import java.util.Map;

public final class UBCryptoProvider extends Provider
{
  private static final String NAME = "UNBOUND";
  private static final double VERSION = 1.0;
  private static final String INFO = "UnboundTech UKC security provider";

  private static final String ENV_SERVERS = "UKC_SERVERS";
  private static final String ENV_PFX = "UKC_PFX";
  private static final String ENV_PFX_PASS = "UKC_PFX_PASS";
  private static final String ENV_CA = "UKC_CA";
  private static final String ENV_CLIENT_NAME = "UKC_CLIENT_NAME";
  private static final String ENV_TEMPLATE_NAME = "UKC_TEMPLATE_NAME";
  private static final String ENV_PARTITION_NAME = "UKC_PARTITION_NAME";
  private static final String ENV_ACTIVATION_CODE = "UKC_ACTIVATION_CODE";

  private static boolean initialized = false;
  private static final int SERVICE_KEY_STORE = 0;
  private static final int SERVICE_RSA_GEN = 1;
  private static final int SERVICE_RSA_IMPORT = 2;
  private static final int SERVICE_EC_GEN = 3;
  private static final int SERVICE_EC_IMPORT = 4;

  private Partition partition;

  public static synchronized void initialize(String[] servers, KeyStore trusted) throws MalformedURLException, KeyStoreException, NoSuchAlgorithmException
  {
    Server.initialize(servers);
    Connection.initialize(trusted);
    initialized = true;
  }

  private static synchronized void initialize(Map<String, String> map) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException
  {
    if ((map == null || map.isEmpty()) && initialized) return;

    String servers = getConfig(map, ENV_SERVERS);
    String ca = getConfig(map, ENV_CA);

    Server.initialize(servers);
    Connection.initialize(ca);
    initialized = true;
  }

  private UBCryptoProvider(String name, double version, String info)
  {
    super(name, version, info);
  }

  public UBCryptoProvider()
  {
    this((String) null);
  }

  public static UBCryptoProvider proxy(Provider parent, String configArg)
  {
    UBCryptoProvider provider = new UBCryptoProvider(parent.getName(), parent.getVersion(), parent.getInfo());
    try
    {
      provider.init(configArg);
    }
    catch (Exception e)
    {
      if (configArg == null) return null; // wait to call for configure
      throw new ProviderException(e);
    }
    return provider;
    // parent should call register(parent)
  }

  public static UBCryptoProvider getInstance(KeyStore pfx, String pfxPass)
  {
    UBCryptoProvider provider = new UBCryptoProvider(NAME, VERSION, INFO);
    try
    {

      UBCryptoProvider.initialize(null);
      provider.partition = Partition.registerPfx(pfx, pfxPass);
      provider.register(provider);

    }
    catch (ProviderException e)
    {
      throw e;
    }
    catch (Exception e)
    {
      throw new ProviderException(e);
    }
    return provider;
  }

  public UBCryptoProvider(String partitionName, String clientName, String templateName, String activationCode)
  {
    super(NAME, VERSION, INFO);
    try
    {

      initialize(null);
      partition = Client.register(partitionName, clientName, templateName, activationCode);
      register(this);

    }
    catch (ProviderException e)
    {
      throw e;
    }
    catch (Exception e)
    {
      throw new ProviderException(e);
    }
  }

  public UBCryptoProvider(String configArg)
  {
    super(NAME, VERSION, INFO);

    boolean hasConfig = configArg!=null;

    if (!hasConfig)
      hasConfig =
        System.getenv(ENV_SERVERS) != null &&
        System.getenv(ENV_CA) != null;

    if (!hasConfig) return;

    try
    {
      init(configArg);
      register(this);
    }
    catch (ProviderException e)
    {
      throw e;
    }
    catch (Exception e)
    {
      throw new ProviderException(e);
    }
  }

  private static String getConfig(Map<String,String> map, String name)
  {
    String result = null;
    if (map!=null) result = map.get(name);
    if (result==null) result = System.getenv(name);
    return result;
  }

  private void init(String configArg) throws CertificateException, NoSuchAlgorithmException, KeyStoreException, IOException, UnrecoverableKeyException
  {
    Log log = Log.func("UBCryptoProvider.init").log("configArg", configArg).end();
    try
    {
      Map<String, String> map = (configArg == null) ? null : Config.readFile(configArg);

      String pfx = getConfig(map, ENV_PFX);
      String pfxPass = getConfig(map, ENV_PFX_PASS);
      String partitionName = getConfig(map, ENV_PARTITION_NAME);
      String clientName = getConfig(map, ENV_CLIENT_NAME);
      String templateName = getConfig(map, ENV_TEMPLATE_NAME);
      String activationCode = getConfig(map, ENV_ACTIVATION_CODE);

      UBCryptoProvider.initialize(map);

      if (pfx==null)
      {
        partition = Client.register(partitionName, clientName, templateName, activationCode);
      }
      else
      {
        partition = Partition.registerPfx(pfx, pfxPass);
      }
    }
    catch (Exception e)
    {
      log.failed(e);
      throw e;
    }
    finally
    {
      log.leave();
    }
  }

  // Since java 9
  public Provider configure(String configArg)
  {
    return new UBCryptoProvider(configArg);
  }

  public Service[] register(Provider provider)
  {
    final String prefix = getClass().getPackage().getName() + ".";
    final String rsaKeyClasses = "java.security.interfaces.RSAPublicKey|java.security.interfaces.RSAPrivateKey";
    provider.put("KeyFactory.RSA", prefix + "RSAKeyFactory");
    provider.put("KeyPairGenerator.RSA", prefix + "RSAKeyPairGenerator");
    provider.put("Alg.Alias.KeyPairGenerator.1.2.840.113549.1.1", "RSA");
    provider.put("Alg.Alias.KeyPairGenerator.OID.1.2.840.113549.1.1", "RSA");

    StringBuffer modes = new StringBuffer();
    appendMode(modes, "PKCS1PADDING");
    appendMode(modes, "NOPADDING");
    appendMode(modes, "OAEPPADDING"
            + "|OAEPWITHSHA1ANDMGF1PADDING"
            + "|OAEPWITHSHA-1ANDMGF1PADDING"
            + "|OAEPWITHSHA-256ANDMGF1PADDING"
            + "|OAEPWITHSHA-384ANDMGF1PADDING"
            + "|OAEPWITHSHA-512ANDMGF1PADDING");

    if (modes.length() > 0)
    {
      provider.put("Cipher.RSA", prefix + "RSACipher");
      provider.put("Cipher.RSA SupportedModes", "ECB");
      provider.put("Cipher.RSA SupportedKeyClasses", rsaKeyClasses);
      provider.put("Cipher.RSA SupportedPaddings", modes.toString());
    }

    provider.put("Signature.NONEwithRSA SupportedKeyClasses", rsaKeyClasses);
    provider.put("Signature.NONEwithRSA", prefix + "RSASignature$NONEwithRSA");

    provider.put("Signature.SHA1withRSA SupportedKeyClasses", rsaKeyClasses);
    provider.put("Signature.SHA1withRSA", prefix + "RSASignature$SHA1withRSA");
    provider.put("Alg.Alias.Signature.1.2.840.113549.1.1.5", "SHA1withRSA");
    provider.put("Alg.Alias.Signature.OID.1.2.840.113549.1.1.5", "SHA1withRSA");
    provider.put("Alg.Alias.Signature.1.3.14.3.2.29", "SHA1withRSA");
    provider.put("Signature.SHA256withRSA SupportedKeyClasses", rsaKeyClasses);
    provider.put("Signature.SHA256withRSA", prefix + "RSASignature$SHA256withRSA");
    provider.put("Alg.Alias.Signature.1.2.840.113549.1.1.11", "SHA256withRSA");
    provider.put("Alg.Alias.Signature.OID.1.2.840.113549.1.1.11", "SHA256withRSA");
    provider.put("Signature.SHA384withRSA SupportedKeyClasses", rsaKeyClasses);
    provider.put("Signature.SHA384withRSA", prefix + "RSASignature$SHA384withRSA");
    provider.put("Alg.Alias.Signature.1.2.840.113549.1.1.12", "SHA384withRSA");
    provider.put("Alg.Alias.Signature.OID.1.2.840.113549.1.1.12", "SHA384withRSA");
    provider.put("Signature.SHA512withRSA SupportedKeyClasses", rsaKeyClasses);
    provider.put("Signature.SHA512withRSA", prefix + "RSASignature$SHA512withRSA");
    provider.put("Alg.Alias.Signature.1.2.840.113549.1.1.13", "SHA512withRSA");
    provider.put("Alg.Alias.Signature.OID.1.2.840.113549.1.1.13", "SHA512withRSA");

    String ecKeyClasses = "java.security.interfaces.ECPublicKey|java.security.interfaces.ECPrivateKey";

    //put("KeyFactory.EC", prefix + "ECKeyFactory");
    //put("KeyPairGenerator.EC", prefix + "ECKeyPairGenerator");
    provider.put("Alg.Alias.KeyFactory.EllipticCurve", "EC");

    provider.put("Signature.NONEwithECDSA", prefix + "ECDSASignature$Raw");
    provider.put("Signature.NONEwithECDSA SupportedKeyClasses", ecKeyClasses);

    provider.put("Signature.SHA1withECDSA", prefix + "ECDSASignature$SHA1");
    provider.put("Signature.SHA1withECDSA SupportedKeyClasses", ecKeyClasses);

    provider.put("Signature.SHA256withECDSA", prefix + "ECDSASignature$SHA256");
    provider.put("Signature.SHA256withECDSA SupportedKeyClasses", ecKeyClasses);

    provider.put("Signature.SHA384withECDSA", prefix + "ECDSASignature$SHA384");
    provider.put("Signature.SHA384withECDSA SupportedKeyClasses", ecKeyClasses);

    provider.put("Signature.SHA512withECDSA", prefix + "ECDSASignature$SHA512");
    provider.put("Signature.SHA512withECDSA SupportedKeyClasses", ecKeyClasses);


    Service[] services = new Service[5];

    services[0] = new UBProviderService(provider, "KeyStore", "PKCS11", prefix + "UBKeyStore", partition, SERVICE_KEY_STORE);
    services[1] = new UBProviderService(provider, "KeyFactory", "RSA", prefix + "RSAKeyFactory", partition, SERVICE_RSA_IMPORT);
    services[2] = new UBProviderService(provider, "KeyPairGenerator", "RSA", prefix + "RSAKeyPairGenerator", partition, SERVICE_RSA_GEN);
    services[3] = new UBProviderService(provider, "KeyFactory", "EC", prefix + "ECKeyFactory", partition, SERVICE_EC_IMPORT);
    services[4] = new UBProviderService(provider, "KeyPairGenerator", "EC", prefix + "ECKeyPairGenerator", partition, SERVICE_EC_GEN);

    if (this == provider)
    {
      for (Service service : services) putService(service);
      return null;
    }

    return services;
  }

  private static void appendMode(StringBuffer buf, String a)
  {
    if (buf.length() > 0) buf.append("|");
    buf.append(a);
  }

  final static class UBProviderService extends Service
  {
    private Partition partition;
    int mode;

    UBProviderService(Provider provider, String type, String algorithm, String className, Partition partition, int mode)
    {
      super(provider, type, algorithm, className, null, null);
      this.partition = partition;
      this.mode = mode;
    }

    @Override
    public boolean supportsParameter(Object param)
    {
      return false;
    }

    @Override
    public Object newInstance(Object param)
    {
      switch (mode)
      {
        case SERVICE_KEY_STORE:
          return partition.keyStore;
        case SERVICE_RSA_GEN:
          return new RSAKeyPairGenerator(partition);
        case SERVICE_RSA_IMPORT:
          return new RSAKeyFactory(partition);
        case SERVICE_EC_GEN:
          return new ECKeyPairGenerator(partition);
        case SERVICE_EC_IMPORT:
          return new ECKeyFactory(partition);
      }
      return null;
    }
  }
}


