package com.unbound.provider;

import com.unbound.common.Log;
import com.unbound.common.crypto.X509;
import com.unbound.provider.kmip.KMIP;
import com.unbound.provider.kmip.attribute.*;
import com.unbound.provider.kmip.request.*;
import com.unbound.provider.kmip.response.*;

import javax.security.auth.x500.X500Principal;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.ProviderException;
import java.security.PublicKey;
import java.security.cert.*;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;

class UBCertificate extends UBObject
{
  X509Certificate x509;

  UBCertificate(Partition partition, long uid, GetAttributesResponse getAttrResp, GetResponse get) throws CertificateException
  {
    super(partition, uid, getAttrResp);
    Log log = Log.func("UBCertificate").logHex("uid", uid).end(); try
    {
      com.unbound.provider.kmip.object.Certificate kmipManagedObject = (com.unbound.provider.kmip.object.Certificate)get.object;
      x509 = X509.get(kmipManagedObject.value);
    }
    catch (Exception e) { log.failed(e); throw e; } finally { log.leave(); }
  }

  UBCertificate(Partition partition, String alias, X509Certificate x509) throws IOException, CertificateEncodingException
  {
    super(partition);

    RequestMessage reqMsg = new RequestMessage();

    RegisterRequest reqRegister = new RegisterRequest();
    reqMsg.batch.add(reqRegister);
    reqRegister.objectType = KMIP.ObjectType.Certificate;
    if (alias!=null)
    {
      reqRegister.template = new TemplateAttribute();
      reqRegister.template.attrs.add(new Name(alias));
    }

    com.unbound.provider.kmip.object.Certificate managedObject = new com.unbound.provider.kmip.object.Certificate();
    managedObject.type = KMIP.CertificateType.X_509;
    managedObject.value = x509.getEncoded();
    reqRegister.object = managedObject;

    ActivateRequest reqActivate = new ActivateRequest();
    reqMsg.batch.add(reqActivate);

    GetAttributesRequest reqGetAttr = new GetAttributesRequest();
    reqMsg.batch.add(reqGetAttr);
    reqGetAttr.names.add("Name");
    reqGetAttr.names.add("Initial Date");

    ResponseMessage respMsg = partition.transmit(reqMsg);

    RegisterResponse respRegister = (RegisterResponse)respMsg.batch.get(0);
    GetAttributesResponse respGetAttr = (GetAttributesResponse)respMsg.batch.get(2);

    uid = strToUid(respRegister.uid);
    name = ((Name)respGetAttr.attrs.get(0)).value;
    initialDate = ((DateAttribute)respGetAttr.attrs.get(1)).value;
    this.x509 = x509;
  }

  boolean match(PublicKey key)
  {
    return x509.getPublicKey().equals(key);
  }

  static UBCertificate locate(Partition partition, String alias) throws CertificateException, InvalidKeySpecException, IOException
  {
    long uid = 0;
    Log log = Log.func("UBCertificate.locate").log("alias", alias).end(); try
    {
      uid = partition.locate(KMIP.ObjectType.Certificate, 0, alias);
      if (uid==0) return null;
      return (UBCertificate) UBObject.read(partition, uid);
    } catch (Exception e) { log.failed(e); throw e; } finally { log.leavePrint().logHex("uid", uid).end(); }
  }

  static UBCertificate locate(Partition partition, X509Certificate cert) throws CertificateException, InvalidKeySpecException, IOException
  {
    long uid = 0;
    Log log = Log.func("UBCertificate.locateByValue").end(); try
    {
      LocateRequest req = locateRequest(KMIP.ObjectType.Certificate, 0, null);
      req.attrs.add(new BytesAttribute(KMIP.Tag.CKA_VALUE, cert.getEncoded()));
      uid = partition.locate(req);
      if (uid==0) return null;
      return (UBCertificate) UBObject.read(partition, uid);
    }
    catch (Exception e) { log.failed(e); throw e; } finally { log.leavePrint().logHex("uid", uid).end(); }
  }

  static UBCertificate locateByKeyUid(Partition partition, long keyUid) throws CertificateException, InvalidKeySpecException, IOException
  {
    long uid = 0;
    Log log = Log.func("UBCertificate.locateByKeyUid").logHex("keyUid", keyUid).end(); try
    {
      LocateRequest req = locateRequest(KMIP.ObjectType.Certificate, 0, null);
      req.attrs.add(new Link(KMIP.LinkType.PrivateKeyLink, uidToStr(keyUid)));
      uid = partition.locate(req);
      if (uid==0) return null;
      return (UBCertificate) UBObject.read(partition, uid);
    }
    catch (Exception e) { log.failed(e); throw e; } finally { log.leavePrint().logHex("uid", uid).end(); }
  }

  private static UBCertificate findBySubject(X500Principal subject, UBObject[] certs)
  {
    for (UBObject object : certs)
    {
      if (object==null) continue;
      UBCertificate cert = (UBCertificate)object;
      if (subject.equals(cert.x509.getSubjectX500Principal()))
      {
        return cert;
      }
    }
    return null;
  }

  Certificate[] getChain(UBObject[] certs)
  {
    ArrayList<X509Certificate> chain = new ArrayList<>();
    chain.add(x509);
    UBCertificate next = this;
    X500Principal subject = x509.getSubjectX500Principal();

    for (;;)
    {
      X500Principal issuer = next.x509.getIssuerX500Principal();
      if (subject.equals(issuer)) break;

      next = findBySubject(issuer, certs);
      if (next==null) break;

      chain.add(next.x509);
    }

    return chain.toArray(new X509Certificate[0]);
  }

  private static UBCertificate findBySubject(Partition partition, X500Principal subject) throws CertificateException, InvalidKeySpecException, IOException
  {
    LocateRequest req = locateRequest(KMIP.ObjectType.Certificate, 0, null);
    req.attrs.add(new BytesAttribute(KMIP.Tag.CKA_SUBJECT, subject.getEncoded()));
    long uid = partition.locate(req);
    if (uid==0) return null;
    return (UBCertificate) UBObject.read(partition, uid);
  }

  Certificate[] getChain() throws CertificateException, InvalidKeySpecException, IOException
  {
    ArrayList<X509Certificate> chain = new ArrayList<>();
    Log log = Log.func("UBCertificate.getChain").logHex("uid", uid).end(); try
    {
      chain.add(x509);
      UBCertificate next = this;
      X500Principal subject = x509.getSubjectX500Principal();

      for (;;)
      {
        X500Principal issuer = next.x509.getIssuerX500Principal();
        if (subject.equals(issuer)) break;

        next = findBySubject(partition, issuer);
        if (next==null) break;

        chain.add(next.x509);
      }

      return chain.toArray(new X509Certificate[0]);
    }
    catch (Exception e) { log.failed(e); throw e; } finally { log.leave(); }
  }

  @Override
  int objectType()
  {
    return KMIP.ObjectType.Certificate;
  }

  static long getKeyUid(X509Certificate x509)
  {
    PublicKey pub = x509.getPublicKey();
    if (pub instanceof RSAPublicKey) return UBRSAPrivateKey.getKeyUid((RSAPublicKey)pub);
    if (pub instanceof ECPublicKey) return UBECPrivateKey.getKeyUid((ECPublicKey)pub);

    throw new ProviderException("Unsupported certificate type");
  }
}
