package com.unbound.provider;

import com.unbound.common.crypto.SystemProvider;
import com.unbound.provider.kmip.KMIP;

import java.io.IOException;
import java.security.*;
import java.security.interfaces.RSAPublicKey;
import java.util.Arrays;

public class RSASignature extends SignatureSpi
{
  private Signature pubSignature = null;
  private UBRSAPrivateKey prvKey = null;
  private int hashBitSize;
  private MessageDigest md = null;
  private byte[] buffer = new byte[501]; // 512 - 11
  private int bufferOffset = 0;

  static String hashBitSizeToName(int hashBitSize) throws InvalidAlgorithmParameterException
  {
    switch (hashBitSize)
    {
      case 0:    return "NONE";
      case 160:  return "SHA1";
      case 256:  return "SHA256";
      case 384:  return "SHA384";
      case 512:  return "SHA512";
    }
    throw new InvalidAlgorithmParameterException("Unsupported hash algorithm");
  }

  static String hashBitSizeToDigestName(int hashBitSize) throws InvalidAlgorithmParameterException
  {
    switch (hashBitSize)
    {
      case 160:  return "SHA-1";
      case 256:  return "SHA-256";
      case 384:  return "SHA-384";
      case 512:  return "SHA-512";
    }
    throw new InvalidAlgorithmParameterException("Unsupported hash algorithm");
  }

  private static int hashBitSizeToKmipAlg(int hashBitSize)
  {
    switch (hashBitSize)
    {
      case 160: return KMIP.DigitalSignatureAlgorithm.SHA_1WithRSAEncryption;
      case 256: return KMIP.DigitalSignatureAlgorithm.SHA_256WithRSAEncryption;
      case 384: return KMIP.DigitalSignatureAlgorithm.SHA_384WithRSAEncryption;
      case 512: return KMIP.DigitalSignatureAlgorithm.SHA_512WithRSAEncryption;
    }
    return 0;
  }

  RSASignature(int hashBitSize)
  {
    this.hashBitSize = hashBitSize;
  }



  // ------------------------- interface ---------------------------------

  @Override
  protected void engineInitVerify(PublicKey publicKey) throws InvalidKeyException
  {
    if (!(publicKey instanceof RSAPublicKey)) throw new InvalidKeyException("Invalid key type");

    try { pubSignature = SystemProvider.Signature.getInstance(hashBitSizeToName(hashBitSize)+"WithRSA"); }
    catch (Throwable e) { throw new InvalidKeyException("engineInitVerify failed"); }
    pubSignature.initVerify(publicKey);
  }

  @Override
  protected void engineInitSign(PrivateKey privateKey) throws InvalidKeyException
  {
    if (!(privateKey instanceof UBRSAPrivateKey)) throw new InvalidKeyException("Invalid key type");
    prvKey = (UBRSAPrivateKey)privateKey;
    bufferOffset = 0;

    try
    {
      md = (hashBitSize==0) ? null : SystemProvider.MessageDigest.getInstance(hashBitSizeToDigestName(hashBitSize));
    }
    catch (InvalidAlgorithmParameterException e) { throw new ProviderException(e); }
  }

  @Override
  protected void engineUpdate(byte b) throws SignatureException
  {
    if (pubSignature!=null)
    {
      pubSignature.update(b);
      return;
    }

    byte[] in = {b};
    engineUpdate(in, 0, 1);
  }

  @Override
  protected void engineUpdate(byte[] in, int inOffset, int inLen) throws SignatureException
  {
    if (pubSignature!=null)
    {
      pubSignature.update(in, inOffset, inLen);
      return;
    }

    if (md==null)
    {
      System.arraycopy(in, inOffset, buffer, bufferOffset, inLen);
      bufferOffset += inLen;
    }
    else md.update(in, inOffset, inLen);
  }

  @Override
  protected byte[] engineSign() throws SignatureException
  {
    byte[] in = (md==null) ? Arrays.copyOfRange(buffer, 0, bufferOffset) : md.digest();
    int kmipAlg = hashBitSizeToKmipAlg(hashBitSize);
    try
    {
      return prvKey.sign(in, KMIP.CryptographicAlgorithm.RSA, kmipAlg);
    }
    catch (IOException e) { throw new ProviderException(e); }
  }

  @Override
  protected boolean engineVerify(byte[] sigBytes) throws SignatureException
  {
    return pubSignature.verify(sigBytes);
  }

  @Override
  protected void engineSetParameter(String param, Object value) throws InvalidParameterException
  {
    throw new UnsupportedOperationException("setParameter() not supported");
  }

  @Override
  protected Object engineGetParameter(String param) throws InvalidParameterException
  {
    throw new UnsupportedOperationException("getParameter() not supported");
  }

  // ----------------------- sub-classes ----------------------
  public static final class NONEwithRSA extends RSASignature
  {
    public NONEwithRSA() { super(0); }
  }

  public static final class SHA1withRSA extends RSASignature
  {
    public SHA1withRSA() { super(160); }
  }

  public static final class SHA256withRSA extends RSASignature
  {
    public SHA256withRSA() { super(256); }
  }

  public static final class SHA384withRSA extends RSASignature
  {
    public SHA384withRSA() { super(384); }
  }

  public static final class SHA512withRSA extends RSASignature
  {
    public SHA512withRSA() { super(512); }
  }

}
