package com.unbound.provider;

import com.unbound.common.crypto.SystemProvider;

import java.security.*;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.*;

public final class RSAKeyFactory extends KeyFactorySpi
{
  private Partition partition;

  RSAKeyFactory(Partition partition)
  {
    this.partition = partition;
  }

// --------------------- interface -----------------

  @Override
  protected PublicKey engineGeneratePublic(KeySpec keySpec) throws InvalidKeySpecException
  {
    return SystemProvider.KeyFactory.getInstance("RSA").generatePublic(keySpec);
  }

  @Override
  protected PrivateKey engineGeneratePrivate(KeySpec keySpec) throws InvalidKeySpecException
  {
    if (keySpec == null) throw new InvalidKeySpecException("keySpec == null");

    if ((keySpec instanceof RSAPrivateCrtKeySpec) || (keySpec instanceof PKCS8EncodedKeySpec))
    {
      try { return new UBRSAPrivateKey(partition, keySpec); }
      catch (Exception e) { throw new InvalidKeySpecException(e); }
    }
    throw new InvalidKeySpecException("Must use RSAPrivateCrtKeySpec or PKCS8EncodedKeySpec; was " + keySpec.getClass().getName());
  }

  @Override
  protected <T extends KeySpec> T engineGetKeySpec(Key key, Class<T> keySpec) throws InvalidKeySpecException
  {
    if ((key == null) || (keySpec == null)) throw new InvalidKeySpecException("key and keySpec must not be null");

    if (key instanceof RSAPublicKey && X509EncodedKeySpec.class.isAssignableFrom(keySpec))
    {
      return SystemProvider.KeyFactory.getInstance("RSA").getKeySpec(key, keySpec);
    }

    if (key instanceof RSAPublicKey && RSAPublicKeySpec.class.isAssignableFrom(keySpec))
    {
        return (T) new RSAPublicKeySpec(((RSAPublicKey)key).getModulus(), ((RSAPublicKey)key).getPublicExponent());
    }

    throw new InvalidKeySpecException("Could not encode key");
  }

  @Override
  protected Key engineTranslateKey(Key key) throws InvalidKeyException
  {
    return key;
  }
}
