package com.unbound.provider;

import com.unbound.common.crypto.EC;
import com.unbound.common.crypto.SystemProvider;
import com.unbound.provider.kmip.KMIP;

import java.io.IOException;
import java.security.*;
import java.security.interfaces.ECPublicKey;
import java.util.Arrays;

public class ECDSASignature extends SignatureSpi
{
  private Signature pubSignature = null;
  private int hashBitSize;
  private MessageDigest md = null;
  private byte[] buffer = new byte[66]; // max curve size
  private int bufferOffset = 0;
  private UBECPrivateKey prvKey = null;

  ECDSASignature(int hashBitSize)
  {
    this.hashBitSize = hashBitSize;
  }

  private static int hashBitSizeToKmipAlg(int hashBitSize)
  {
    switch (hashBitSize)
    {
      case 160: return KMIP.DigitalSignatureAlgorithm.ECDSAWithSHA_1;
      case 256: return KMIP.DigitalSignatureAlgorithm.ECDSAWithSHA256;
      case 384: return KMIP.DigitalSignatureAlgorithm.ECDSAWithSHA384;
      case 512: return KMIP.DigitalSignatureAlgorithm.ECDSAWithSHA512;
    }
    return 0;
  }

// ------------------------------ interface ----------------------

  @Override
  protected void engineInitVerify(PublicKey publicKey) throws InvalidKeyException
  {
    if (!(publicKey instanceof ECPublicKey)) throw new InvalidKeyException("Invalid key type");

    try { pubSignature = Signature.getInstance(RSASignature.hashBitSizeToName(hashBitSize)+"withECDSA", "SunEC"); }
    catch (Exception e) { throw new InvalidKeyException("engineInitVerify failed"); }
    pubSignature.initVerify(publicKey);
  }

  @Override
  protected void engineInitSign(PrivateKey privateKey) throws InvalidKeyException
  {
    if (!(privateKey instanceof UBECPrivateKey)) throw new InvalidKeyException("Invalid key type");
    prvKey = (UBECPrivateKey)privateKey;
    bufferOffset = 0;

    try
    {
      md = (hashBitSize==0) ? null : SystemProvider.MessageDigest.getInstance(RSASignature.hashBitSizeToDigestName(hashBitSize));
    }
    catch (Exception e) { throw new ProviderException(e); }
  }

  @Override
  protected void engineUpdate(byte b) throws SignatureException
  {
    if (pubSignature!=null)
    {
      pubSignature.update(b);
      return;
    }

    byte[] in = {b};
    engineUpdate(in, 0, 1);
  }

  @Override
  protected void engineUpdate(byte[] in, int inOffset, int inLen) throws SignatureException
  {
    if (pubSignature!=null)
    {
      pubSignature.update(in, inOffset, inLen);
      return;
    }

    if (md==null)
    {
      int size = prvKey.size();
      if (bufferOffset+inLen > size)  inLen = size - bufferOffset;
      System.arraycopy(in, inOffset, buffer, bufferOffset, inLen);
      bufferOffset += inLen;
    }
    else md.update(in, inOffset, inLen);
  }

  @Override
  protected byte[] engineSign() throws SignatureException
  {
    EC.Curve curve = prvKey.getCurve();

    byte[] in;
    if (md==null)
    {
      in = Arrays.copyOfRange(buffer, 0, bufferOffset);
    }
    else
    {
      in = md.digest();
      if (in.length>curve.size) in = Arrays.copyOfRange(in,0, curve.size);
    }
    int kmipAlg = hashBitSizeToKmipAlg(hashBitSize);
    try
    {
      byte[] bin = prvKey.sign(in, KMIP.CryptographicAlgorithm.EC, kmipAlg);
      return curve.sigBinToDer(bin);
    }
    catch (IOException e) { throw new ProviderException(e); }
  }

  @Override
  protected boolean engineVerify(byte[] sigBytes) throws SignatureException
  {
    return pubSignature.verify(sigBytes);
  }

  @Override
  protected void engineSetParameter(String param, Object value) throws InvalidParameterException
  {
    throw new UnsupportedOperationException("setParameter() not supported");
  }

  @Override
  protected Object engineGetParameter(String param) throws InvalidParameterException
  {
    throw new UnsupportedOperationException("getParameter() not supported");
  }

  public static final class Raw extends ECDSASignature
  {
    public Raw() { super(0); }
  }

  public static final class SHA1 extends ECDSASignature
  {
    public SHA1() { super(160); }
  }

  public static final class SHA256 extends ECDSASignature
  {
    public SHA256() { super(256); }
  }

  public static final class SHA384 extends ECDSASignature
  {
    public SHA384() { super(384); }
  }

  public static final class SHA512 extends ECDSASignature
  {
    public SHA512() { super(512); }
  }

}
