package com.dyadicsec.provider;

import com.dyadicsec.cryptoki.CK;
import com.dyadicsec.pkcs11.*;

import java.math.BigInteger;
import java.security.*;
import java.security.spec.ECPoint;
import java.util.Arrays;

public class SchnorrSignature extends SignatureSpi {
    private final int mechanismType = CK.DYCKM_SCHNORR;
    protected ECPrivateKey prvKey = null;
    protected Session session = null;
    private final byte[] buffer = new byte[32];
    private int bufferOffset = 0;

    public SchnorrSignature()
    {
    }

    @Override
    protected void engineInitVerify(PublicKey publicKey) throws InvalidKeyException
    {
        java.security.interfaces.ECPublicKey sunPubKey = null;
        if (publicKey instanceof ECPublicKey) sunPubKey = ((ECPublicKey) publicKey).getSoftwareKey();
        else if (publicKey instanceof java.security.interfaces.ECPublicKey) sunPubKey = (java.security.interfaces.ECPublicKey)publicKey;
        else throw new InvalidKeyException("Invalid key type");

        closeSession();
        try {
            Slot slot = DYCryptoProvider.getDefaultKeyStore().getSlot();
            CKECPublicKey pkcsPubKey = CKECPublicKey.create(slot, null, null, ECCurve.find("secp256k1"), sunPubKey.getW());
            session = pkcsPubKey.verifyInit(new CK_MECHANISM(CK.DYCKM_SCHNORR));
        }
        catch (CKException e) {
            throw new InvalidKeyException(e);
        }
    }

    protected void checkInit()
    {
        if (session!=null) return;
        bufferOffset = 0;
        try
        {
            session = prvKey.pkcs11Key.signInit(new CK_MECHANISM(mechanismType));
        }
        catch (CKException e) { throw new ProviderException(e); }
    }

    @Override
    protected void engineInitSign(PrivateKey privateKey) throws InvalidKeyException
    {
        if (privateKey instanceof ECPrivateKey) prvKey = (ECPrivateKey) privateKey;
        else throw new InvalidKeyException("Invalid key type");

        try { prvKey.save(); }
        catch (KeyStoreException e) { throw new InvalidKeyException(e); }

        closeSession();
        checkInit();
    }

    @Override
    protected void engineUpdate(byte b) throws SignatureException
    {
        byte[] in = {b};
        engineUpdate(in, 0, 1);
    }

    @Override
    protected void engineUpdate(byte[] in, int inOffset, int inLen) throws SignatureException
    {
        checkInit();

        if (bufferOffset+inLen > buffer.length) throw new SignatureException("Invalid input length");
        System.arraycopy(in, inOffset, buffer, bufferOffset, inLen);
        bufferOffset += inLen;
    }

    @Override
    protected byte[] engineSign() throws SignatureException
    {
        try
        {
            return session.sign(buffer, 0, bufferOffset, 64);
        }
        catch (CKException e) { throw new SignatureException(e); }
        finally { closeSession(); }
    }

    @Override
    protected boolean engineVerify(byte[] sigBytes) throws SignatureException
    {
        try { return session.verify(buffer, sigBytes); }
        catch (CKException e) { throw new SignatureException(e); }
        finally { closeSession(); }
//        return pubSignature.verify(sigBytes);
    }

    @Override
    @Deprecated
    protected void engineSetParameter(String param, Object value) throws InvalidParameterException
    {
        throw new UnsupportedOperationException("setParameter() not supported");
    }

    @Override
    @Deprecated
    protected Object engineGetParameter(String param) throws InvalidParameterException
    {
        throw new UnsupportedOperationException("getParameter() not supported");
    }

    private void closeSession()
    {
        if (session!=null) session.close();
        session = null;
        bufferOffset = 0;
    }

}
