package com.dyadicsec.provider;

import com.dyadicsec.cryptoki.CK;
import com.dyadicsec.pkcs11.CKException;
import com.dyadicsec.pkcs11.CK_MECHANISM;
import com.dyadicsec.pkcs11.Session;

import java.io.ByteArrayOutputStream;
import java.security.*;

public class EDDSASignature extends SignatureSpi {

    protected Session session = null;
    private ByteArrayOutputStream buffer = new ByteArrayOutputStream();
    int mechanismType;

    EDDSASignature(int mechanismType)
    {
        this.mechanismType = mechanismType;
    }

    private void closeSession()
    {
        if (session!=null) session.close();
        session = null;
    }

    private void checkInit() throws SignatureException
    {
        if (session == null) throw new SignatureException();
    }

    @Override
    protected void engineInitSign(PrivateKey privateKey) throws InvalidKeyException
    {
        if (!(privateKey instanceof EDDSAPrivateKey)) throw new InvalidKeyException("Invalid key type");
        EDDSAPrivateKey eddsaPrivateKey = (EDDSAPrivateKey) privateKey;

        try { eddsaPrivateKey.save(); }
        catch (KeyStoreException e) { throw new InvalidKeyException(e); }
        closeSession();
        try {
            session = eddsaPrivateKey.pkcs11Key.signInit(new CK_MECHANISM(mechanismType));
        }
        catch (CKException e) { throw new InvalidKeyException(e); }
        buffer.reset();
    }

    @Override
    protected byte[] engineSign() throws SignatureException
    {
        checkInit();
        try {
            return session.sign(buffer.toByteArray(), 0);
        }
        catch (CKException e) { throw new SignatureException(e); }
        finally { closeSession(); }
    }

    @Override
    protected void engineInitVerify(PublicKey publicKey) throws InvalidKeyException
    {
        if (!(publicKey instanceof EDDSAPublicKey)) throw new InvalidKeyException("Invalid key type");

        closeSession();
        try {
            session = ((EDDSAPublicKey) publicKey).pkcs11Key.verifyInit(new CK_MECHANISM(mechanismType));
        }
        catch (CKException e) {
            throw new InvalidKeyException(e);
        }
        if (buffer==null) buffer = new ByteArrayOutputStream(); else buffer.reset();
    }

    @Override
    protected boolean engineVerify(byte[] sigBytes) throws SignatureException
    {
        checkInit();
        try {
            return session.verify(buffer.toByteArray(), sigBytes);
        }
        catch (CKException e) { throw new SignatureException(e); }
        finally { closeSession(); }
    }

    @Override
    protected void engineUpdate(byte b) throws SignatureException
    {
        byte[] in = {b};
        engineUpdate(in, 0, 1);
    }

    @Override
    protected void engineUpdate(byte[] in, int inOffset, int inLen) throws SignatureException
    {
        checkInit();
        buffer.write(in, inOffset, inLen);
    }

    @Override
    protected Object engineGetParameter(String param) throws InvalidParameterException
    {
        throw new UnsupportedOperationException("getParameter() not supported");
    }

    @Override
    protected void engineSetParameter(String param, Object value) throws InvalidParameterException
    {
        throw new UnsupportedOperationException("setParameter() not supported");
    }

    public static final class Raw extends EDDSASignature
    {
        public Raw() { super(CK.DYCKM_EDDSA); }
    }

}
