package com.dyadicsec.provider;

import java.security.spec.*;
import java.security.KeyFactorySpi;
import java.security.Key;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.InvalidKeyException;
import java.security.KeyStoreException;

/**
 * Created by valery.osheter on 19-Apr-16.
 */
public final class ECKeyFactory extends KeyFactorySpi
{
    @Override
    protected PublicKey engineGeneratePublic(KeySpec keySpec) throws InvalidKeySpecException
    {
        if (keySpec == null) throw new InvalidKeySpecException("keySpec == null");

        KeyParameters keyParams = null;
        KeyFactorySpec keyFactorySpec = null;
        if (keySpec instanceof KeyFactorySpec)
        {
            keyFactorySpec = (KeyFactorySpec) keySpec;
            keySpec = keyFactorySpec.original;
            keyParams = keyFactorySpec.params;
        }


        if (keySpec instanceof ECPublicKeySpec || keySpec instanceof X509EncodedKeySpec)
        {
            try { return new ECPublicKey(keySpec, keyParams); }
            catch (NoSuchAlgorithmException e) { throw new InvalidKeySpecException(e); }
            catch (NoSuchProviderException e) { throw new InvalidKeySpecException(e); }
        }
        throw new InvalidKeySpecException("Must use ECPublicKeySpec or X509EncodedKeySpec; was " + keySpec.getClass().getName());
    }

    @Override
    protected PrivateKey engineGeneratePrivate(KeySpec keySpec) throws InvalidKeySpecException
    {
        if (keySpec == null) throw new InvalidKeySpecException("keySpec == null");

        KeyParameters keyParams = null;
        KeyFactorySpec keyFactorySpec = null;
        if (keySpec instanceof KeyFactorySpec)
        {
            keyFactorySpec = (KeyFactorySpec) keySpec;
            keySpec = keyFactorySpec.original;
            keyParams = keyFactorySpec.params;
        }

        if (keySpec instanceof ECPrivateKeySpec || keySpec instanceof PKCS8EncodedKeySpec)
        {
            try { return new ECPrivateKey().initForImport(keyParams, keySpec); }
            catch (Exception e) { throw new InvalidKeySpecException(e); }
        }
        throw new InvalidKeySpecException("Must use ECPrivateKeySpec or PKCS8EncodedKeySpec; was " + keySpec.getClass().getName());
    }

    @Override
    @SuppressWarnings("unchecked")
    protected <T extends KeySpec> T engineGetKeySpec(Key key, Class<T> keySpec) throws InvalidKeySpecException
    {
        if ((key == null) || (keySpec == null)) throw new InvalidKeySpecException("key and keySpec must not be null");

        if (key instanceof ECPublicKey && X509EncodedKeySpec.class.isAssignableFrom(keySpec))
        {
            try { return KeyFactory.getInstance("EC", "SunEC").getKeySpec(key, keySpec); }
            catch (NoSuchAlgorithmException e) { throw new InvalidKeySpecException("Could not encode key", e); }
            catch (NoSuchProviderException e) { throw new InvalidKeySpecException("Could not encode key", e); }
        }

        if (key instanceof ECPublicKey && ECPublicKeySpec.class.isAssignableFrom(keySpec))
        {
            return (T) new ECPublicKeySpec(((ECPublicKey)key).getW(), ((ECPublicKey)key).getParams());
        }

        throw new InvalidKeySpecException("Could not encode key");
    }

    @Override
    protected Key engineTranslateKey(Key key) throws InvalidKeyException
    {
        if (key == null) throw new InvalidKeyException("CKKey must not be null");
        if (key instanceof ECPrivateKey) return key;
        if (key instanceof ECPublicKey) return key;

        if (key instanceof java.security.interfaces.ECPublicKey) return new ECPublicKey((java.security.interfaces.ECPublicKey)key);
        if (key instanceof java.security.interfaces.ECPrivateKey)
        {
            try { return new ECPrivateKey().initForImport((java.security.interfaces.ECPrivateKey) key); }
            catch (KeyStoreException e) { throw new InvalidKeyException(e); }
        }
        throw new InvalidKeyException("CKKey must be instance of CKPublicKey or CKPrivateKey");
    }

}
