package com.dyadicsec.pkcs11;

import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.util.Arrays;

/**
 * Created by saar.peer on 18-Jul-16.
 */
public final class Utils
{
    public static long id2uid(byte[] id)
    {
        if (id.length!=9) return 0;
        if (id[0]!=0x00) return 0;
        long uid = 0;
        for (int i=1; i<9; i++)
        {
            uid <<= 8;
            uid += (id[i] & 0xff);
        }
        return uid;
    }

    public static byte[] uid2id(long uid)
    {
        byte[] id = new byte[9];
        id[0] = 0x00;
        for (int i=8; i>1; i--)
        {
            id[i] = (byte)uid;
            uid >>= 8;
        }
        return id;
    }

    private static boolean isPrintableChar(char ch)
    {
        if ((ch >= 'a' && ch <= 'z') || (ch >= 'A' && ch <= 'Z') || (ch >= '0' && ch <= '9')) return true;
        switch (ch)
        {
            case ' ':       /* space */
            case '\'':      /* apostrophe */
            case '(':       /* left paren */
            case ')':       /* right paren */
            case '+':       /* plus */
            case ',':       /* comma */
            case '-':       /* hyphen */
            case '.':       /* period */
            case '/':       /* slash */
            case ':':       /* colon */
            case '=':       /* equals */
            case '?':       /* question mark */
                return true;
        }
        return false;
    }

    private static char hex2Char(int x)
    {
        x &= 15;
        return x<10 ? (char)('0'+x) : (char)('a'+x-10);
    }

    public static String id2name(byte[] id)
    {
        if (id.length==0) return "0x000000000000000000"; // 0x00 + UID=0

        boolean printable = true;
        for (int i=0; printable && i<id.length; i++) printable = isPrintableChar((char)id[i]);

        if (printable)
        {
            try { return new String(id, "UTF-8"); }
            catch (UnsupportedEncodingException e) { return "0x000000000000000000"; }
        }

        char[] name = new char[2+id.length*2];
        int o = 0;
        name[o++] = '0';
        name[o++] = 'x';
        for (int i=0; i<id.length; i++)
        {
            name[o++] = hex2Char(id[i]>>4);
            name[o++] = hex2Char(id[i]);
        }

        return String.valueOf(name);
    }

    private static byte char2Hex(char c)
    {
        if (c>='0' && c<='9') return (byte)(c-'0');
        if (c>='a' && c<='f') return (byte)(c-'a'+10);
        if (c>='A' && c<='F') return (byte)(c-'A'+10);
        return 0;
    }

    public static byte[] name2id(String name)
    {
        if (name==null) return new byte[0];
        int nameSize = name==null ? 0 : name.length();

        if (nameSize>=4 && name.charAt(0)=='0' && name.charAt(1)=='x')
        {
            int idSize = (nameSize-2)/2;
            byte[] id = new byte[idSize];
            int p = 2;
            for (int i=0; i<idSize; i++)
            {
                char hi = name.charAt(p++);
                char lo = name.charAt(p++);
                id[i] =  (byte)((char2Hex(hi) << 4) + char2Hex(lo));
            }
            return id;
        }

        try { return name.getBytes("UTF-8"); }
        catch (UnsupportedEncodingException e) { return new byte[0]; }
    }

    static public byte[] bigInt2Bytes(BigInteger b, int size)
    {
        byte[] bytes = b.toByteArray();
        if (bytes==null) return null;
        if (size==0)
        {
            if (bytes[0]==0) return Arrays.copyOfRange(bytes, 1, bytes.length);
            return bytes;
        }

        if (bytes.length==size) return bytes;
        if (bytes.length==size+1 && bytes[0]==0) return Arrays.copyOfRange(bytes, 1, size+1);
        if (size>bytes.length)
        {
            byte[] result = new byte[size];
            System.arraycopy(bytes, 0, result, size-bytes.length, bytes.length);
            return result;
        }

        return null; // !!!
    }

    public static int bigIntByteSize(BigInteger b)
    {
        return b.bitLength() / 8;
    }

    static public void uidToBytes(long uid, byte[] out, int outOffset)
    {
        out[outOffset + 0] = (byte) (uid >> 56);
        out[outOffset + 1] = (byte) (uid >> 48);
        out[outOffset + 2] = (byte) (uid >> 40);
        out[outOffset + 3] = (byte) (uid >> 32);
        out[outOffset + 4] = (byte) (uid >> 24);
        out[outOffset + 5] = (byte) (uid >> 16);
        out[outOffset + 6] = (byte) (uid >> 8);
        out[outOffset + 7] = (byte) (uid);
    }

    static public byte[] uidToBytes(long uid)
    {
        byte[] out = new byte[8];
        uidToBytes(uid, out, 0);
        return out;
    }

    private static long byteToULong(byte b) {
        return ((long) b) & 0xff;
    }

    static public long bytesToUID(byte[] in, int inOffset)
    {
        return
                (byteToULong(in[inOffset+0]) << 56) |
                        (byteToULong(in[inOffset+1]) << 48) |
                        (byteToULong(in[inOffset+2]) << 40) |
                        (byteToULong(in[inOffset+3]) << 32) |
                        (byteToULong(in[inOffset+4]) << 24) |
                        (byteToULong(in[inOffset+5]) << 16) |
                        (byteToULong(in[inOffset+6]) << 8)  |
                        (byteToULong(in[inOffset+7]));
    }

    static public long bytesToUID(byte[] in)
    {
        return bytesToUID(in, 0);
    }

}
