package com.dyadicsec.pkcs11;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Map;
import static com.dyadicsec.cryptoki.CK.*;
import static com.dyadicsec.pkcs11.DYCK_FPE_PARAMS.DYCK_FPE_STRING;

/**
 * Created by valery.osheter on 22-Jun-17.
 */
public final class CKSecretKey extends CKKey
{
    int bitSize = 0;

    CKSecretKey() { clazz= CKO_SECRET_KEY; }

    void prepareReadTemplate(Map<Integer, CK_ATTRIBUTE> template)
    {
        super.prepareReadTemplate(template);
        addReadTemplate(template, CKA_KEY_TYPE);
        addReadTemplate(template, CKA_VALUE_LEN);
    }

    void saveReadTemplate(Map<Integer, CK_ATTRIBUTE> template) throws CKException
    {
        super.saveReadTemplate(template);
        keyType = template.get(CKA_KEY_TYPE).toInt();
        bitSize = template.get(CKA_VALUE_LEN).toInt()*8;
    }

    public byte[] getValue() throws CKException
    {
        return getAttributeValue(CKA_VALUE);
    }

    public int getBitSize() throws CKException
    {
        if (bitSize==0) read();
        return bitSize;
    }

    public static CK_ATTRIBUTE[] getUnwrapTemplate(String name, Policy policy, int keyType)
    {
        if (policy==null) policy = new Policy();
        return new CK_ATTRIBUTE[]
                {
                        new CK_ATTRIBUTE(CKA_TOKEN, policy.cka_token),
                        new CK_ATTRIBUTE(CKA_PRIVATE, true),
                        new CK_ATTRIBUTE(CKA_CLASS, CKO_SECRET_KEY),
                        new CK_ATTRIBUTE(CKA_KEY_TYPE, keyType),
                        new CK_ATTRIBUTE(CKA_DERIVE, policy.cka_derive),
                        new CK_ATTRIBUTE(CKA_SENSITIVE, policy.cka_sensitive),
                        new CK_ATTRIBUTE(CKA_EXTRACTABLE, policy.cka_extractable),
                        new CK_ATTRIBUTE(CKA_ENCRYPT, policy.cka_encrypt),
                        new CK_ATTRIBUTE(CKA_DECRYPT, policy.cka_decrypt),
                        new CK_ATTRIBUTE(CKA_SIGN, policy.cka_sign),
                        new CK_ATTRIBUTE(CKA_VERIFY, policy.cka_verify),
                        new CK_ATTRIBUTE(CKA_WRAP, policy.cka_wrap),
                        new CK_ATTRIBUTE(CKA_UNWRAP, policy.cka_unwrap),
                        new CK_ATTRIBUTE(CKA_TRUSTED, policy.cka_trusted),
                };
    }

    public static CKSecretKey generate(Slot slot, String name, Policy policy, int keyType, int bitSize) throws CKException
    {
        if (policy==null) policy = new Policy();
        CKSecretKey key = new CKSecretKey();
        if (keyType== CKK_DES3) bitSize = 192;

        CK_ATTRIBUTE[] t =
                {
                        new CK_ATTRIBUTE(CKA_TOKEN, policy.cka_token),
                        new CK_ATTRIBUTE(CKA_CLASS, CKO_SECRET_KEY),
                        new CK_ATTRIBUTE(CKA_KEY_TYPE, keyType),
                        new CK_ATTRIBUTE(CKA_SENSITIVE, policy.cka_sensitive),
                        new CK_ATTRIBUTE(CKA_EXTRACTABLE, policy.cka_extractable),
                        new CK_ATTRIBUTE(CKA_ENCRYPT, policy.cka_encrypt),
                        new CK_ATTRIBUTE(CKA_DECRYPT, policy.cka_decrypt),
                        new CK_ATTRIBUTE(CKA_SIGN, policy.cka_sign),
                        new CK_ATTRIBUTE(CKA_VERIFY, policy.cka_verify),
                        new CK_ATTRIBUTE(CKA_WRAP, policy.cka_wrap),
                        new CK_ATTRIBUTE(CKA_UNWRAP, policy.cka_unwrap),
                        new CK_ATTRIBUTE(CKA_TRUSTED, policy.cka_trusted),
                        new CK_ATTRIBUTE(CKA_VALUE_LEN, bitSize/8),
                        new CK_ATTRIBUTE(CKA_ID, Utils.name2id(name)),
                };

        int genMechanism = getGenerateMechanism(keyType);

        key.generateKey(slot, genMechanism, t);
        key.bitSize = bitSize;
        key.keyType = keyType;
        key.policy = policy;
        key.name = name;
        return key;
    }

    public static CKSecretKey create(Slot slot, String name, Policy policy, int keyType, byte[] value) throws CKException
    {
        if (policy==null) policy = new Policy();
        CKSecretKey key = new CKSecretKey();

        CK_ATTRIBUTE[] t =
                {
                        new CK_ATTRIBUTE(CKA_TOKEN, policy.cka_token),
                        new CK_ATTRIBUTE(CKA_CLASS, CKO_SECRET_KEY),
                        new CK_ATTRIBUTE(CKA_KEY_TYPE, keyType),
                        new CK_ATTRIBUTE(CKA_SENSITIVE, policy.cka_sensitive),
                        new CK_ATTRIBUTE(CKA_EXTRACTABLE, policy.cka_extractable),
                        new CK_ATTRIBUTE(CKA_ENCRYPT, policy.cka_encrypt),
                        new CK_ATTRIBUTE(CKA_DECRYPT, policy.cka_decrypt),
                        new CK_ATTRIBUTE(CKA_SIGN, policy.cka_sign),
                        new CK_ATTRIBUTE(CKA_VERIFY, policy.cka_verify),
                        new CK_ATTRIBUTE(CKA_WRAP, policy.cka_wrap),
                        new CK_ATTRIBUTE(CKA_UNWRAP, policy.cka_unwrap),
                        new CK_ATTRIBUTE(CKA_TRUSTED, policy.cka_trusted),
                        new CK_ATTRIBUTE(CKA_VALUE, value),
                        new CK_ATTRIBUTE(CKA_ID, Utils.name2id(name)),
                };
        key.create(slot, t);
        key.bitSize = value.length * 8;
        key.keyType = keyType;
        key.policy = policy;
        key.name = name;
        return key;
    }

    public static CKSecretKey find(Slot slot, int keyType, String name)
    {
        return (CKSecretKey) CKObject.find(slot, CKO_SECRET_KEY, keyType, name);
    }

    public static CKSecretKey find(Slot slot, String name)
    {
        return find(slot, -1, name);
    }

    public static CKSecretKey find(Slot slot, long uid)
    {
        return CKObject.find(slot, CKSecretKey.class, uid);
    }

    public static ArrayList<CKSecretKey> list(Slot slot, int keyType)
    {
        return CKObject.list(slot, CKSecretKey.class, CKO_SECRET_KEY, keyType);
    }

    public static ArrayList<CKSecretKey> list(Slot slot) { return list(slot, -1); }

    public byte[] hmacSha256(byte[] in) throws  CKException
    {
        return sign(new CK_MECHANISM(CKM_SHA256_HMAC), in, 32);
    }

    public byte[] encryptFPE(int mode, char[] format, int maxLen, byte[] in) throws  CKException
    {
        int outLen = 0;
        if (mode==DYCK_FPE_PARAMS.DYCK_FPE_CREDIT_CARD || mode==DYCK_FPE_PARAMS.DYCK_FPE_US_PHONE || mode==DYCK_FPE_PARAMS.DYCK_FPE_SSN)
        {
            if (format!=null && format.length!=0) outLen = in.length;
            else outLen = in.length;
        }
        return encrypt(new DYCK_FPE_PARAMS(mode, format, maxLen), in, outLen);
    }

    public byte[] decryptFPE(int mode, char[] format, byte[] in) throws  CKException
    {
        int outLen = 0;
        if (mode==DYCK_FPE_PARAMS.DYCK_FPE_CREDIT_CARD || mode==DYCK_FPE_PARAMS.DYCK_FPE_US_PHONE || mode==DYCK_FPE_PARAMS.DYCK_FPE_SSN)
        {
            if (format!=null && format.length!=0) outLen = in.length;
            else outLen = in.length;
        }
        return decrypt(new DYCK_FPE_PARAMS(mode, format, 0), in, outLen);
    }

    public byte[] encryptSPE(byte[] in, int bits) throws  CKException
    {
        return encrypt(new DYCK_SPE_PARAMS(bits), in, in.length);
    }

    public byte[] decryptSPE(byte[] in, int bits) throws  CKException
    {
        return decrypt(new DYCK_SPE_PARAMS(bits), in, in.length);
    }

    public byte[] encryptOPE(byte[] in) throws  CKException
    {
        return encrypt(new CK_MECHANISM(DYCKM_OPE), in, in.length+4);
    }

    public byte[] decryptOPE(byte[] in) throws  CKException
    {
        if (in.length<5) throw new CKException("decryptOPE", CKR_ENCRYPTED_DATA_LEN_RANGE);
        return decrypt(new CK_MECHANISM(DYCKM_OPE), in, in.length-4);
    }

    public String encryptStringFPE(String in, String format) throws CKException
    {
        try
        {
            byte[] inBuf = in.getBytes("UTF-16BE");
            byte[] outBuf = encryptFPE(DYCK_FPE_STRING, format.toCharArray(), 0, inBuf);
            return new String(outBuf, "UTF-16BE");
        }
        catch (UnsupportedEncodingException e) { throw new CKException("Unsupported encoding", CKR_DATA_INVALID); }
    }

    public String decryptStringFPE(String in, String format) throws CKException
    {
        try
        {
            byte[] inBuf = in.getBytes("UTF-16BE");
            byte[] outBuf = decryptFPE(DYCK_FPE_STRING, format.toCharArray(), inBuf);
            return new String(outBuf, "UTF-16BE");
        }
        catch (UnsupportedEncodingException e) { throw new CKException("Unsupported encoding", CKR_ENCRYPTED_DATA_INVALID); }
    }
}

