package com.dyadicsec.pkcs11;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Map;
import static com.dyadicsec.cryptoki.CK.*;

import static com.dyadicsec.pkcs11.Utils.bigInt2Bytes;

/**
 * Created by valery.osheter on 22-Jun-17.
 */
public final class CKRSAPublicKey extends CKPublicKey
{
    CKRSAPublicKey()
    {
        keyType = CKK_RSA;
    }

    int bitsSize = 0;
    BigInteger N = null;
    BigInteger E = null;

    void prepareReadTemplate(Map<Integer, CK_ATTRIBUTE> template)
    {
        super.prepareReadTemplate(template);
        addReadTemplate(template, CKA_PUBLIC_EXPONENT);
        addReadTemplate(template, CKA_MODULUS);
    }

    void saveReadTemplate(Map<Integer, CK_ATTRIBUTE> template) throws CKException
    {
        super.saveReadTemplate(template);
        byte[] NBuf = template.get(CKA_MODULUS).getValue();
        E = new BigInteger(1,template.get(CKA_PUBLIC_EXPONENT).getValue());
        N = new BigInteger(1,NBuf);
        bitsSize = NBuf.length*8;
    }

    public BigInteger getE() throws CKException
    {
        if (E==null) read();
        return E;
    }

    public BigInteger getN() throws CKException
    {
        if (N==null) read();
        return N;
    }

    public static CKRSAPublicKey create(Slot slot, String name, Policy policy, BigInteger N, BigInteger E) throws CKException
    {
        byte[] NBuf = bigInt2Bytes(N, 0);
        CKRSAPublicKey key = new CKRSAPublicKey();
        if (policy==null) policy = CKPublicKey.getDefaultPolicy();
        CK_ATTRIBUTE[] t =
                {
                        new CK_ATTRIBUTE(CKA_TOKEN, policy.cka_token),
                        new CK_ATTRIBUTE(CKA_CLASS, CKO_PUBLIC_KEY),
                        new CK_ATTRIBUTE(CKA_KEY_TYPE, CKK_RSA),
                        new CK_ATTRIBUTE(CKA_ENCRYPT, policy.cka_encrypt),
                        new CK_ATTRIBUTE(CKA_VERIFY, policy.cka_verify),
                        new CK_ATTRIBUTE(CKA_WRAP, policy.cka_wrap),
                        new CK_ATTRIBUTE(CKA_TRUSTED, policy.cka_trusted),
                        new CK_ATTRIBUTE(CKA_MODULUS, NBuf),
                        new CK_ATTRIBUTE(CKA_PUBLIC_EXPONENT, E.toByteArray()),
                        new CK_ATTRIBUTE(CKA_ID, Utils.name2id(name)),
                };

        key.create(slot, t);
        key.policy = policy;
        key.E = E;
        key.N = N;
        key.name = name;
        key.bitsSize = NBuf.length*8;
        return key;
    }

    public int getBitsSize() throws CKException
    {
        if (bitsSize==0) read();
        return bitsSize;
    }

    public static CKRSAPublicKey find(Slot slot, String name)
    {
        return (CKRSAPublicKey) CKObject.find(slot, CKO_PUBLIC_KEY, CKK_RSA, name);
    }

    public static CKRSAPublicKey find(Slot slot, long uid)
    {
        return CKObject.find(slot, CKRSAPublicKey.class, uid);
    }

    public static ArrayList<CKRSAPublicKey> list(Slot slot)
    {
        return CKObject.list(slot, CKRSAPublicKey.class, CKO_PUBLIC_KEY, CKK_RSA);
    }

    public byte[] encrypt(CK_MECHANISM mechanism, byte[] in) throws CKException
    {
        return super.encrypt(mechanism, in, getBitsSize()/8);
    }

    public byte[] encrypt(int mechanism, byte[] in) throws CKException
    {
        return encrypt(new CK_MECHANISM(mechanism), in);
    }

    public byte[] wrap(CK_MECHANISM mechanism, CKKey key) throws CKException
    {
        return super.wrap(mechanism, key, getBitsSize()/8);
    }

    public byte[] wrap(int mechanism, CKKey key) throws CKException
    {
        return wrap(new CK_MECHANISM(mechanism), key);
    }

    public boolean verify(int mechanism, byte[] in, byte[] signature) throws CKException
    {
        return verify(new CK_MECHANISM(mechanism), in, signature);
    }


}
