package com.dyadicsec.pkcs11;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import static com.dyadicsec.cryptoki.CK.*;

/**
 * Created by valery.osheter on 22-Jun-17.
 */
public abstract class CKObject
{
    int handle = 0;
    Slot slot = null;
    long uid = 0;
    long replacedUID = -1;
    int clazz = -1;
    String name;
    Policy policy = null;

    CKObject() {}

    CKObject(int handle) { this.handle = handle; }

    void addReadTemplate(Map<Integer, CK_ATTRIBUTE> template, int attribute)
    {
        template.put(attribute, new CK_ATTRIBUTE(attribute));
    }

    void prepareReadTemplate(Map<Integer, CK_ATTRIBUTE> template)
    {
        addReadTemplate(template, CKA_TOKEN);
        addReadTemplate(template, DYCKA_UID);
        addReadTemplate(template, CKA_ID);
        addReadTemplate(template, KMIP_REPLACED_UID);
    }

    void saveReadTemplate(Map<Integer, CK_ATTRIBUTE> template) throws CKException
    {
        policy.setToken(template.get(CKA_TOKEN).toBool());
        uid = template.get(DYCKA_UID).toLong();
        name = Utils.id2name(template.get(CKA_ID).getValue());
        replacedUID = template.get(KMIP_REPLACED_UID).toLong();
    }

    void read() throws CKException
    {
        HashMap<Integer, CK_ATTRIBUTE> template = new HashMap<Integer, CK_ATTRIBUTE>();
        prepareReadTemplate(template);
        CK_ATTRIBUTE[] array = template.values().toArray(new CK_ATTRIBUTE[0]);
        getAttributeValue(array);
        if (policy==null) policy = new Policy();
        saveReadTemplate(template);
    }

    public int getHandle() { return handle; }

    public Slot getSlot() { return slot; }

    public void destroy() throws CKException
    {
        slot.destroyObject(handle);
    }

    public void create(Slot slot, CK_ATTRIBUTE[] template) throws CKException
    {
        if (slot==null) slot = Slot.getDefault();
        handle = slot.createObject(template);
        this.slot = slot;
    }

    public Policy getPolicy() throws CKException
    {
        if (policy==null)
        {
            policy = new Policy();
            if (handle!=0) read();
        }
        return policy;
    }


    public int getAttributeSize(int attribute) throws CKException
    {
        return slot.getAttributeSize(handle, attribute);
    }

    public void getAttributeValue(CK_ATTRIBUTE[] template) throws CKException
    {
        slot.getAttributeValue(handle, template);
    }

    public void setAttributeValue(CK_ATTRIBUTE[] template) throws CKException
    {
        slot.setAttributeValue(handle, template);
    }

    public byte[] getAttributeValue(int type) throws CKException
    {
        CK_ATTRIBUTE[] t = new CK_ATTRIBUTE[] { new CK_ATTRIBUTE(type) };
        slot.getAttributeValue(handle, t);
        return t[0].getValue();
    }

    public int getAttributeValueInt(int type) throws CKException
    {
        CK_ATTRIBUTE[] t = new CK_ATTRIBUTE[] { new CK_ATTRIBUTE(type) };
        slot.getAttributeValue(handle, t);
        return t[0].toInt();
    }

    public boolean getAttributeValueBool(int type) throws CKException
    {
        CK_ATTRIBUTE[] t = new CK_ATTRIBUTE[] { new CK_ATTRIBUTE(type) };
        slot.getAttributeValue(handle, t);
        return t[0].toBool();
    }

    public long getAttributeValueLong(int type) throws CKException
    {
        CK_ATTRIBUTE[] t = new CK_ATTRIBUTE[] { new CK_ATTRIBUTE(type) };
        slot.getAttributeValue(handle, t);
        return t[0].toLong();
    }

    public long getUID() throws CKException
    {
        if (uid==0) read();
        return uid;
    }

    public long getReplacedUID() throws CKException
    {
        if (replacedUID==-1) read();
        return replacedUID;
    }

    public int getClazz()
    {
        return clazz;
    }

    public String getName() throws CKException
    {
        return Utils.id2name(getAttributeValue(CKA_ID));
    }

    public void setName(String name) throws CKException
    {
        setAttributeValue(new CK_ATTRIBUTE[] { new CK_ATTRIBUTE(CKA_ID, Utils.name2id(name)) });
    }

    public static <T extends CKObject> T find(Slot slot, Class<T> c, CK_ATTRIBUTE[] template)
    {
        if (slot==null) return Slot.findObjectInAllSlots(c, template);
        else return slot.findObject(c, template);
    }

    public static <T extends CKObject> T find(Slot slot, Class<T> c, long uid)
    {
        if (slot==null) return Slot.findObjectInAllSlots(c, uid);
        else return slot.findObject(c, uid);
    }

    public static CKObject find(Slot slot, int clazz, int keyType, String name)
    {
        if (slot==null) return Slot.findObjectInAllSlots(clazz, keyType, name);
        else return slot.findObject(clazz, keyType, name);
    }

    public static <T extends CKObject>ArrayList<T> list(Slot slot, Class<T> c, CK_ATTRIBUTE[] template)
    {
        if (slot==null) return Slot.findObjectsInAllSlots(c, template);
        else return slot.findObjects(c, template);
    }

    public static <T extends CKObject>ArrayList<T> list(Slot slot, Class<T> c, int clazz, int keyType)
    {
        if (slot==null) return Slot.findObjectsInAllSlots(c, clazz, keyType);
        else return slot.findObjects(c, clazz, keyType);
    }

    public static ArrayList<CKObject> list(Slot slot, int clazz, int keyType)
    {
        if (slot==null) return Slot.findObjectsInAllSlots(clazz, keyType);
        else return slot.findObjects(clazz, keyType);
    }
}
