package com.dyadicsec.pkcs11;

import java.util.Map;
import static com.dyadicsec.cryptoki.CK.*;

/**
 * Created by valery.osheter on 22-Jun-17.
 */
public abstract class CKKey extends CKObject
{
    int keyType = -1;

    void prepareReadTemplate(Map<Integer, CK_ATTRIBUTE> template)
    {
        super.prepareReadTemplate(template);
        addReadTemplate(template, CKA_DERIVE);
    }

    void saveReadTemplate(Map<Integer, CK_ATTRIBUTE> template) throws CKException
    {
        super.saveReadTemplate(template);
        policy.setDerive(template.get(CKA_DERIVE).toBool());
    }

    void generateKeyPair(Slot slot, int mechType, CK_ATTRIBUTE[] pubTemplate, CK_ATTRIBUTE[] prvTemplate) throws CKException
    {
        if (slot==null) slot = Slot.getDefault();
        this.handle = slot.generateKeyPair(mechType, pubTemplate, prvTemplate);
        this.slot = slot;
    }

    void generateKey(Slot slot, int mechType, CK_ATTRIBUTE[] template) throws CKException
    {
        if (slot==null) slot = Slot.getDefault();
        this.handle = slot.generateKey(mechType, template);
        this.slot = slot;
    }

    public int getKeyType() throws CKException
    {
        if (keyType==-1) read();
        return keyType;
    }

    public <T extends CKKey> T derive(Class<T> c, CK_MECHANISM mechanism, CK_ATTRIBUTE[] t) throws CKException
    {
        return slot.newObject(c, slot.deriveKey(mechanism, handle, t));
    }

    public <T extends CKKey> T unwrap(Class<T> c, CK_MECHANISM mechanism, byte[] in, CK_ATTRIBUTE[] template) throws CKException
    {
        return unwrap(c, mechanism, in, 0, in.length, template);
    }

    public <T extends CKKey> T unwrap(Class<T> c, CK_MECHANISM mechanism, byte[] in, int inOffset, int inLen, CK_ATTRIBUTE[] template) throws CKException
    {
        return slot.newObject(c, slot.unwrapKey(mechanism, handle, in, inOffset, inLen, template));
    }

    public byte[] wrap(CK_MECHANISM mechanism, CKKey key, int outLen)  throws CKException
    {
        return slot.persistentSession.wrap(mechanism, this, key, outLen);
    }

    public Session encryptInit(CK_MECHANISM mechanism) throws CKException
    {
        Session session = slot.getSession();
        try { session.encryptInit(mechanism, this); }
        catch (CKException e) { session.close(); throw e; }
        return session;
    }

    public Session decryptInit(CK_MECHANISM mechanism) throws CKException
    {
        Session session = slot.getSession();
        try { session.decryptInit(mechanism, this); }
        catch (CKException e) { session.close(); throw e; }
        return session;
    }

    public Session signInit(CK_MECHANISM mechanism) throws CKException
    {
        Session session = slot.getSession();
        try { session.signInit(mechanism, this); }
        catch (CKException e) { session.close(); throw e; }
        return session;
    }

    public Session verifyInit(CK_MECHANISM mechanism) throws CKException
    {
        Session session = slot.getSession();
        try { session.verifyInit(mechanism, this);}
        catch (CKException e) { session.close(); throw e; }
        return session;
    }

    public byte[] encrypt(CK_MECHANISM mechanism, byte[] in, int outLen) throws CKException
    {
        return Session.encrypt(mechanism, this, in, outLen);
    }

    public byte[] decrypt(CK_MECHANISM mechanism, byte[] in, int outLen) throws CKException
    {
        return Session.decrypt(mechanism, this, in, outLen);
    }

    public byte[] sign(CK_MECHANISM mechanism, byte[] in, int outLen) throws CKException
    {
        return Session.sign(mechanism, this, in, outLen);
    }

    public boolean verify(CK_MECHANISM mechanism, byte[] in, byte[] signature) throws CKException
    {
        return Session.verify(mechanism, this, in, signature);
    }

    public static int getGenerateMechanism(int keyType)
    {
        switch (keyType)
        {
            case CKK_RSA: return CKM_RSA_PKCS_KEY_PAIR_GEN;
            case CKK_EC: return CKM_EC_KEY_PAIR_GEN;
            case DYCKK_LIMA: return DYCKM_LIMA_KEY_GEN;
            case DYCKK_ADV_PRF: return DYCKM_PRF_KEY_GEN;
            case DYCKK_ADV_PASSWORD: return DYCKM_PASSWORD_KEY_GEN;
            case CKK_AES: return CKM_AES_KEY_GEN;
            case DYCKK_AES_XTS: return DYCKM_AES_XTS_KEY_GEN;
            case DYCKK_AES_SIV: return DYCKM_AES_SIV_KEY_GEN;
            case CKK_DES3: return CKM_DES3_KEY_GEN;
            case CKK_GENERIC_SECRET: return CKM_GENERIC_SECRET_KEY_GEN;
            case DYCKK_EDDSA: return DYCKM_EDDSA_KEY_GEN;
        }
        return CK_UNAVAILABLE_INFORMATION;
    }

    public CKKey rekey() throws CKException
    {
        int genMechanism = getGenerateMechanism(keyType);

        CK_ATTRIBUTE[] t =
                {
                        new CK_ATTRIBUTE(CKA_TOKEN, true),
                        new CK_ATTRIBUTE(CKA_PRIVATE, true),
                        new CK_ATTRIBUTE(CKA_CLASS, clazz),
                        new CK_ATTRIBUTE(CKA_KEY_TYPE, getKeyType()),
                        new CK_ATTRIBUTE(KMIP_REPLACED_UID, getUID()),
                };

        int newHandle = slot.generateKey(genMechanism, t);
        return (CKKey)slot.newObject(newHandle, clazz, getKeyType());
    }
}
