package com.dyadicsec.pkcs11;

import java.io.IOException;
import java.security.spec.ECPoint;
import java.util.ArrayList;
import java.util.Map;
import static com.dyadicsec.cryptoki.CK.*;

/**
 * Created by valery.osheter on 22-Jun-17.
 */
public class CKECPublicKey extends CKPublicKey
{
    CKECPublicKey()
    {
        keyType = CKK_EC;
    }

    ECCurve curve = null;
    ECPoint point = null;

    void prepareReadTemplate(Map<Integer, CK_ATTRIBUTE> template)
    {
        super.prepareReadTemplate(template);
        addReadTemplate(template, CKA_EC_PARAMS);
        addReadTemplate(template, CKA_EC_POINT);
    }

    void saveReadTemplate(Map<Integer, CK_ATTRIBUTE> template) throws CKException
    {
        super.saveReadTemplate(template);

        curve = ECCurve.find(template.get(CKA_EC_PARAMS).getValue());
        if (curve==null) throw new CKException("Unsupported EC curve", 0);

        try { point = curve.derDecodePoint(template.get(CKA_EC_POINT).getValue()); }
        catch (IOException e) { throw new CKException(e, "Can't decode ECPoint", 0); }
    }

    public static CKECPublicKey create(Slot slot, String name, Policy policy, ECCurve curve, ECPoint point) throws CKException
    {
        CKECPublicKey key = new CKECPublicKey();
        if (policy==null) policy = CKPublicKey.getDefaultPolicy();

        CK_ATTRIBUTE[] t =
                {
                        new CK_ATTRIBUTE(CKA_TOKEN, policy.cka_token),
                        new CK_ATTRIBUTE(CKA_CLASS, CKO_PUBLIC_KEY),
                        new CK_ATTRIBUTE(CKA_KEY_TYPE, CKK_EC),
                        new CK_ATTRIBUTE(CKA_VERIFY, policy.cka_verify),
                        new CK_ATTRIBUTE(CKA_EC_PARAMS, curve.getOidBin()),
                        new CK_ATTRIBUTE(CKA_EC_POINT, curve.derEncodePoint(point)),
                        new CK_ATTRIBUTE(CKA_ID, Utils.name2id(name)),
                };

        key.create(slot, t);
        key.policy = policy;
        key.curve = curve;
        key.point = point;
        key.name = name;
        return key;
    }

    public ECCurve getCurve() throws CKException
    {
        if (curve==null) read();
        return curve;
    }

    public ECPoint getPoint() throws CKException
    {
        if (point==null) read();
        return point;
    }

    public static CKECPublicKey find(Slot slot, String name)
    {
        return (CKECPublicKey) CKObject.find(slot, CKO_PUBLIC_KEY, CKK_EC, name);
    }

    public static CKECPublicKey find(Slot slot, long uid)
    {
        return CKObject.find(slot, CKECPublicKey.class, uid);
    }

    public static ArrayList<CKECPublicKey> list(Slot slot)
    {
        return CKObject.list(slot, CKECPublicKey.class, CKO_PUBLIC_KEY, CKK_EC);
    }

    public boolean verify(int mechanism, byte[] in, byte[] signature) throws CKException
    {
        return verify(new CK_MECHANISM(mechanism), in, signature);
    }
}
