package com.dyadicsec.pkcs11;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;
import static com.dyadicsec.cryptoki.CK.*;


/**
 * Created by valery.osheter on 22-Jun-17.
 */
public final class CKCertificate extends CKObject
{
    byte[] value = null;
    X509Certificate cert = null;
    long privateKeyUID = 0;

    CKCertificate() { clazz= CKO_CERTIFICATE; }

    void prepareReadTemplate(Map<Integer, CK_ATTRIBUTE> template)
    {
        super.prepareReadTemplate(template);
        addReadTemplate(template, CKA_VALUE);
        addReadTemplate(template, CKA_CERTIFICATE_CATEGORY);
        addReadTemplate(template, KMIP_PRIVATE_KEY_UID);
    }

    void saveReadTemplate(Map<Integer, CK_ATTRIBUTE> template) throws CKException
    {
        super.saveReadTemplate(template);
        value = template.get(CKA_VALUE).getValue();
        policy.cka_trusted = template.get(CKA_CERTIFICATE_CATEGORY).toInt()==2;
        privateKeyUID = template.get(KMIP_PRIVATE_KEY_UID).toLong();

        policy.cka_extractable = true;
        policy.cka_sensitive = policy.cka_private =
        policy.cka_encrypt = policy.cka_decrypt =
        policy.cka_sign = policy.cka_verify =
        policy.cka_wrap = policy.cka_unwrap =
        policy.cka_derive = false;
    }

    public byte[] getValue() throws CKException
    {
        if (value==null) read();
        return value;
    }

    public X509Certificate getX509() throws CKException, CertificateException
    {
        if (cert==null)
        {
            byte[] value = getValue();
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            cert = (X509Certificate)cf.generateCertificate(new ByteArrayInputStream(value));
        }
        return cert;
    }

    public static CKCertificate create(Slot slot, String name, Policy policy, byte[] value) throws CKException, CertificateException, CertificateEncodingException
    {
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        InputStream in = new ByteArrayInputStream(value);
        X509Certificate cert = (X509Certificate)cf.generateCertificate(in);
        return create(slot, name, policy, cert);
    }

    public static CKCertificate create(Slot slot, String name, Policy policy, X509Certificate cert) throws CKException, CertificateEncodingException
    {
        if (policy==null) policy = new Policy();

        CKCertificate certificate = new CKCertificate();
        CK_ATTRIBUTE[] t = new CK_ATTRIBUTE[]
            {
                new CK_ATTRIBUTE(CKA_TOKEN, policy.cka_token),
                new CK_ATTRIBUTE(CKA_CLASS, CKO_CERTIFICATE),
                new CK_ATTRIBUTE(CKA_CERTIFICATE_TYPE, 0),
                new CK_ATTRIBUTE(CKA_SUBJECT, cert.getSubjectX500Principal().getEncoded()),
                new CK_ATTRIBUTE(CKA_ISSUER, cert.getIssuerX500Principal().getEncoded()),
                new CK_ATTRIBUTE(CKA_SERIAL_NUMBER, cert.getSerialNumber().toByteArray()),
                new CK_ATTRIBUTE(CKA_VALUE, cert.getEncoded()),
                new CK_ATTRIBUTE(CKA_ID, Utils.name2id(name)),
            };

        certificate.create(slot, t);
        certificate.policy = policy;
        certificate.name = name;
        return certificate;
    }

    public static CKCertificate find(Slot slot, String name)
    {
        return (CKCertificate) CKObject.find(slot, CKO_CERTIFICATE, -1, name);
    }

    public static CKCertificate find(Slot slot, long uid)
    {
        return CKObject.find(slot, CKCertificate.class, uid);
    }

    public static ArrayList<CKCertificate> list(Slot slot)
    {
        return CKObject.list(slot, CKCertificate.class, CKO_CERTIFICATE, -1);
    }

    public static ArrayList<CKCertificate> findCertsByPrivateKeyUID(Slot slot, long uid)
    {
      CKCertificate c = CKObject.find(slot, CKCertificate.class, ~uid);

      CK_ATTRIBUTE[] t = new CK_ATTRIBUTE[]
      {
        new CK_ATTRIBUTE(CKA_TOKEN, true),
        new CK_ATTRIBUTE(CKA_CLASS, CKO_CERTIFICATE),
        new CK_ATTRIBUTE(KMIP_PRIVATE_KEY_UID, uid),
      };
      ArrayList<CKCertificate> list = CKObject.list(slot, CKCertificate.class, t);

      if (c!=null)
      {
          for (int i=0; i<list.size(); i++)
          {
            if (list.get(i).handle==c.handle) { c = null; break; }
          }
          if (c!=null) list.add(c);
      }
      return list;
    }

    public static CKCertificate findCertByPrivateKeyUID(Slot slot, long uid)
    {
      ArrayList<CKCertificate> list = findCertsByPrivateKeyUID(slot, uid);

      int n = list.size();
      if (n==0) return null;
      if (n==1) return list.get(0);
      CKCertificate best = null;
      Date bestDate = null;
      for (int i=0; i<n; i++)
      {
          CKCertificate c = list.get(i);
          Date date = null;
          try { date = c.getX509().getNotAfter(); }
          catch (CKException e) { }
          catch (CertificateException e) { }
          if (bestDate==null)
          {
              bestDate = date;
              best = c;
              continue;
          }
          if (date==null)
          {
              continue;
          }
          if (date.after(bestDate))
          {
              bestDate = date;
              best = c;
          }
      }
      return best;
    }

    public long getPrivateKeyUID() throws CKException
    {
        if (privateKeyUID==0) read();
        return privateKeyUID;
    }

}
