package com.dyadicsec.advapi;


import com.dyadicsec.pkcs11.*;

import java.io.UnsupportedEncodingException;
import java.util.Arrays;

import static com.dyadicsec.cryptoki.CK.*;

/**
 * This class includes the methods for application level encryption, see <a href="https://www.unboundtech.com/docs/UKC/UKC_Developers_Guide/HTML/Content/Products/UKC-EKM/UKC_Developers_Guide/Application_Level_Encryption.htm" target="_blank">Application-Level Encryption in the UKC Developers Guide</a> for more information.
 */
public final class SDEKey
{
    /**
     * Use this session key purpose for one way encryption (PRF)
     */
    public static final int PURPOSE_ONE_WAY = 0;
    /**
     * Use this session key purpose for generic type preserving encryption (besides String)
     */
    public static final int PURPOSE_SP_ENC = -1;
    /**
     * Use this session key purpose for order preserving encryption
     */
    public static final int PURPOSE_OP_ENC = -2;
    /**
     * Use this session key purpose for email format preserving encryption
     */
    public static final int PURPOSE_EMAIL_ENC = -3;
    /**
     * Use this session key purpose for credit card number format preserving encryption
     */
    public static final int PURPOSE_CREDIT_CARD_ENC = -4;
    /**
     * Use this session key purpose for US phone format preserving encryption
     */
    public static final int PURPOSE_US_PHONE_ENC = -5;
    /**
     * Use this session key purpose for SSN format preserving encryption
     */
    public static final int PURPOSE_SSN_ENC = -6;
    /**
     * Use this session key purpose for String format preserving encryption (type and size)
     */
    public static final int PURPOSE_STRING_ENC = -7;

    protected CKPRFKey prfKey = null;
    private SDEKey(CKPRFKey prfKey)
    {
        this.prfKey = prfKey;
    }

    private static SDEKey find(Slot slot, String name)
    {
        if (slot==null) return null;
        CKPRFKey prfKey = CKPRFKey.find(slot, name);
        if (prfKey==null) return null;
        return new SDEKey(prfKey);
    }

    private static SDEKey find(Slot slot, long uld)
    {
        if (slot==null) return null;
        CKPRFKey prfKey = CKPRFKey.find(slot, uld);
        if (prfKey==null) return null;
        return new SDEKey(prfKey);
    }

    /**
     * Find an application level encryption key, located by its name
     * @param name The key name
     * @return The key handle or null if the key is not found
     */
    public static SDEKey findKey(String name)
    {
        return find(Slot.getDefault(), name);
    }

    /**
     * Find an application level encryption key in a specific EKM partition.
     * The key is located based on its name
     * @param slotName The partition name
     * @param name The key name
     * @return The key handle or null if the key is not found
     */
    public static SDEKey findKey(String slotName, String name)
    {
        Slot slot = Slot.find(slotName);
        if (slot==null) return null;
        return find(slot, name);
    }

    /**
     * Get the unique identifier of a key, can be used in future time to decrypt with a specific key version
     * @return The unique ID of the key
     * @throws CKException In case of operation error
     */
    public long getUID() throws CKException
    {
        return prfKey.getUID();
    }

    /**
     * Return the previous key used for encryption in case Re-Key was used
     * @return The previous key handle or null if no previous key exists
     */
    public SDEKey getPreviousKey()
    {
        long replacedUID = 0;
        try { replacedUID = prfKey.getReplacedUID(); }
        catch (CKException e) { return null; }
        return find(prfKey.getSlot(), replacedUID);
    }

    /**
     * Find an application level encryption key, located by its UID
     * @param uid The key UID
     * @return The key handle or null if the key is not found
     */
    public static SDEKey findKey(long uid)
    {
        return find(Slot.getDefault(), uid);
    }

    /**
     * Find an application level encryption key in a specific EKM partition.
     * The key is located based on its UID
     * @param slotName The partition name
     * @param uid The key UID
     * @return The key handle or null if the key is not found
     */
    public static SDEKey findKey(String slotName, long uid)
    {
        Slot slot = Slot.find(slotName);
        if (slot==null) return null;
        return find(slot, uid);
    }

    /**
     * Generic encryption function, implements strong AES-GCM none-deterministic encryption and also takes care
     * of key versioning by appending key UID to the cipher text. This encryption mode is not size preserving
     * @param aad Additional Authentication Data, can be used to further authenticate the cipher text,
     *            see reference at https://en.wikipedia.org/wiki/Galois/Counter_Mode
     * @param in The data to encrypt
     * @return Encrypted cipher text
     * @throws SecurityException In case of operation error
     */
    public byte[] encrypt(byte[] aad, byte[] in) throws SecurityException
    {
        try
        {
            byte[] temp = prfKey.encrypt(aad, in);
            byte[] out = new byte[8+temp.length];
            Utils.uidToBytes(getUID(), out, 0);
            System.arraycopy(temp, 0, out, 8, temp.length);
            return out;
        }
        catch (CKException e) { throw new SecurityException(e); }
    }

    /**
     * Generic decryption function, implements strong AES-GCM none-deterministic encryption and also takes care
     * of key versioning by appending key UID to the cipher text. This encryption mode is not size preserving
     * @param aad Additional Authentication Data, can be used to further authenticate the cipher text,
     *            see reference at https://en.wikipedia.org/wiki/Galois/Counter_Mode
     * @param in The data to encrypt
     * @return Encrypted cipher text
     * @throws SecurityException In case of operation error
     */
    public byte[] decrypt(byte[] aad, byte[] in) throws SecurityException
    {
        try
        {
            if (in.length<8) throw new SecurityException("Invalid encrypted data length");
            long uid = Utils.bytesToUID(in, 0);
            SDEKey sdeKey = SDEKey.find(prfKey.getSlot(), uid);
            if (sdeKey==null) throw new SecurityException("Key not found");
            return sdeKey.prfKey.decrypt(aad, Arrays.copyOfRange(in, 8, in.length));
        }
        catch (CKException e) { throw new SecurityException(e); }
    }

    private SDESessionKey generateSessionKey(int purpose, byte[] tweak) throws SecurityException
    {
        int keyType = (purpose==PURPOSE_ONE_WAY) ? CKK_GENERIC_SECRET : CKK_AES;
        try { return new SDESessionKey(this, purpose, prfKey.derive(purpose, tweak, keyType, 256)); }
        catch (CKException e) { throw new SecurityException(e); }
    }

    /**
     * Generate a session key which can be used for multiple encryption operations
     * Use the proper purpose per the values and encryption modes which you will use this key with
     * @param purpose Determines the type and mode of encryption to be used with this session key
     * @param tweak The tweak associated with this session key
     * @return The derived session key
     * @throws SecurityException In case of operation error
     */
    public SDESessionKey generateSessionKey(int purpose, String tweak) throws SecurityException
    {
        try { return generateSessionKey(purpose, SDEUtils.generateTweak(tweak)); }
        catch (UnsupportedEncodingException e) { throw new SecurityException(e); }
    }

    /**
     * Creates a unique searchable token from a byte array
     * @param tweak The tweak used for the process
     * @param data The input data
     * @return Searchable token of size 16 bytes
     * @throws SecurityException In case of encryption error
     */
    public byte[] encryptPRF(String tweak, byte[] data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_ONE_WAY, tweak);
            return sessionKey.encryptPRF(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a byte array
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt, the length of the array should be even or at least 16 bytes
     * @return The encrypted value, the size of the encrypted data equals to the size of the input
     * @throws SecurityException In case of encryption error
     * @throws IllegalArgumentException In case data length is odd and length is smaller than 16
     */
    public byte[] encryptTypePreserving(String tweak, byte[] data) throws SecurityException,
            IllegalArgumentException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt a byte array
     * @param tweak The tweak used for encryption
     * @param encData The encrypted data
     * @return Decrryped byte array
     * @throws SecurityException In case of decryption error
     */
    public byte[] decryptTypePreserving(String tweak, byte[] encData) throws SecurityException,
            IllegalArgumentException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a string value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public String encryptTypePreserving(String tweak, String data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_STRING_ENC, tweak);
            return sessionKey.encryptTypePreserving(data, true);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed string value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return String value in plain
     * @throws SecurityException In case of decryption error
     */
    public String decryptTypePreserving(String tweak, String encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_STRING_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData, true);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a string value in order preserving form
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @param size Maximum size of values that should be compared with this value
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public String encryptOrderPreserving(String tweak, String data, int size) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_OP_ENC, tweak);
            return sessionKey.encryptOrderPreserving(data, size);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt a string value encrypted with order preserving encryption
     * @param tweak The tweak used for the decryption
     * @param encDataStr The encrypted value
     * @return String value in plain
     * @throws SecurityException In case of decryption error
     */
    public String decryptOrderPreserving(String tweak, String encDataStr) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_OP_ENC, tweak);
            return sessionKey.decryptOrderPreserving(encDataStr);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a long value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public long encryptTypePreserving(String tweak, long data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed long value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return long value in plain
     * @throws SecurityException In case of decryption error
     */
    public long decryptTypePreserving(String tweak, long encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a integer value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public int encryptTypePreserving(String tweak, int data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed integer value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return long value in plain
     * @throws SecurityException In case of decryption error
     */
    public int decryptTypePreserving(String tweak, int encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt an integer value in order preserving form, encrypted value is of type long
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public long encryptOrderPreserving(String tweak, int data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_OP_ENC, tweak);
            return sessionKey.encryptOrderPreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an integer value encrypted with order preserving encryption
     * @param tweak The tweak used for the decryption
     * @param encData The encrypted value
     * @return Integer value in plain
     * @throws SecurityException In case of decryption error
     */
    public int decryptOrderPreserving(String tweak, long encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_OP_ENC, tweak);
            return sessionKey.decryptOrderPreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a short value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public short encryptTypePreserving(String tweak, short data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed short value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return String value in plain
     * @throws SecurityException In case of decryption error
     */
    public short decryptTypePreserving(String tweak, short encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }


    /**
     * Encrypt a short value in order preserving form, the return value is of type long
     * @param tweak The tweak used for the encryption
     * @param data The value to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public long encryptOrderPreserving(String tweak, short data) throws SecurityException
    {
        return encryptOrderPreserving(tweak, (int) data);
    }

    /**
     * Encrypt a float value
     * @param tweak The tweak used for the encryption
     * @param data The value to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public float encryptTypePreserving(String tweak, float data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }


    /**
     * Decrypt an encryptyed float value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return Float value in plain
     * @throws SecurityException In case of decryption error
     */
    public float decryptTypePreserving(String tweak, float encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a double value
     * @param tweak The tweak used for the encryption
     * @param data The value to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public double encryptTypePreserving(String tweak, double data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed double value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return Double value in plain
     * @throws SecurityException In case of decryption error
     */
    public double decryptTypePreserving(String tweak, double encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a Date value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public java.sql.Date encryptTypePreserving(String tweak, java.sql.Date data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }


    /**
     * Decrypt an encryptyed Date value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return Date value in plain
     * @throws SecurityException In case of decryption error
     */
    public java.sql.Date decryptTypePreserving(String tweak, java.sql.Date encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a Time value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public java.sql.Time encryptTypePreserving(String tweak, java.sql.Time data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed Time value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return Time value in plain
     * @throws SecurityException In case of decryption error
     */
    public java.sql.Time decryptTypePreserving(String tweak, java.sql.Time encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a Timestamp value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public java.sql.Timestamp encryptTypePreserving(String tweak, java.sql.Timestamp data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed Timestamp value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return Timestamp value in plain
     * @throws SecurityException In case of decryption error
     */
    public java.sql.Timestamp decryptTypePreserving(String tweak, java.sql.Timestamp encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a Timestamp value in order preserving form, encrypted value is of string type
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public String encryptOrderPreserving(String tweak, java.sql.Timestamp data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_OP_ENC, tweak);
            return sessionKey.encryptOrderPreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt a Timestamp value encrypted with order preserving encryption
     * @param tweak The tweak used for the decryption
     * @param encDataStr The encrypted value
     * @return Timestamp value in plain
     * @throws SecurityException In case of decryption error
     */
    public java.sql.Timestamp decryptOrderPreservingTS(String tweak, String encDataStr) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_OP_ENC, tweak);
            return sessionKey.decryptOrderPreservingTS(encDataStr);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a boolean value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public boolean encryptTypePreserving(String tweak, boolean data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed boolean value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return Boolean value in plain
     * @throws SecurityException In case of decryption error
     */
    public boolean decryptTypePreserving(String tweak, boolean encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a Blob value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public java.sql.Blob encryptTypePreserving(String tweak, java.sql.Blob data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed Blob value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return Blob value in plain
     * @throws SecurityException In case of decryption error
     */
    public java.sql.Blob decryptTypePreserving(String tweak, java.sql.Blob encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a Clob value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public java.sql.Clob encryptTypePreserving(String tweak, java.sql.Clob data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed Clob value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return Clob value in plain
     * @throws SecurityException In case of decryption error
     */
    public java.sql.Clob decryptTypePreserving(String tweak, java.sql.Clob encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a BigDecimal value
     * @param tweak The tweak used for the encryption
     * @param data The data to encrypt
     * @return Encrypted value
     * @throws SecurityException In case of encryption error
     */
    public java.math.BigDecimal encryptTypePreserving(String tweak, java.math.BigDecimal data) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.encryptTypePreserving(data);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encryptyed BigDecimal value
     * @param tweak The tweak used for the process
     * @param encData The encrypted value
     * @return BigDecimal value in plain
     * @throws SecurityException In case of decryption error
     */
    public java.math.BigDecimal decryptTypePreserving(String tweak, java.math.BigDecimal encData) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SP_ENC, tweak);
            return sessionKey.decryptTypePreserving(encData);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt an email address in format preserving form, the encrypted value is also a legitimate email address
     * @param tweak The tweak used for the encryption
     * @param in Email address to encrypt
     * @param maxSize Maximum size of all email addresses encrypted
     * @return The encrypted value
     * @throws SecurityException In case of encryption error
     */
    public String encryptEMailAddress(String tweak, String in, int maxSize) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_EMAIL_ENC, tweak);
            return sessionKey.encryptEMailAddress(in, maxSize);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encrypted email address, the encrypted value is also an email address
     * @param tweak The tweak used for the decryption
     * @param in The encrypted value
     * @return Original plain email address
     * @throws SecurityException In case of decryption error
     */
    public String decryptEMailAddress(String tweak, String in) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_EMAIL_ENC, tweak);
            return sessionKey.decryptEMailAddress(in);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a credit card number in format preserving form, the encrypted value is also a legitimate
     * credit card number
     * @param tweak The tweak used for the encryption
     * @param in Credit card number to encrypt
     * @return The encrypted value
     * @throws SecurityException In case of encryption error
     */
    public String encryptCreditCard(String tweak, String in) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_CREDIT_CARD_ENC, tweak);
            return sessionKey.encryptCreditCard(in);
        }
        finally { sessionKey.destroy(); }
    }

    public String encryptCreditCard(String tweak, String in, String format) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_CREDIT_CARD_ENC, tweak);
            return sessionKey.encryptCreditCard(in, format);
        }
        finally { sessionKey.destroy(); }
    }
    /**
     * Decrypt a credit card number, the encrypted value is also a credit card number
     * @param tweak The tweak used for the decryption
     * @param in The encrypted value
     * @return Original plain email address
     * @throws SecurityException In case of decryption error
     */
    public String decryptCreditCard(String tweak, String in) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_CREDIT_CARD_ENC, tweak);
            return sessionKey.decryptCreditCard(in);
        }
        finally { sessionKey.destroy(); }
    }

    public String decryptCreditCard(String tweak, String in, String format) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_CREDIT_CARD_ENC, tweak);
            return sessionKey.decryptCreditCard(in, format);
        }
        finally { sessionKey.destroy(); }
    }
    /**
     * Encrypt a US phone in format preserving form, the encrypted value is also a legitimate US phone number
     * @param tweak The tweak used for the encryption
     * @param in Email address to encrypt
     * @param format The output format, e.g. "###-###-####"
     * @return The encrypted value
     * @throws SecurityException In case of encryption error
     */
    public String encryptUSPhone(final String tweak, final String in, final String format) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_US_PHONE_ENC, tweak);
            return sessionKey.encryptUSPhone(in, format);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encrypted US phone number, the encrypted value is also a US phone number
     * @param tweak The tweak used for the decryption
     * @param in The encrypted value
     * @param format The output format, e.g. "###-###-####"
     * @return Original plain US phone number
     * @throws SecurityException In case of decryption error
     */
    public String decryptUSPhone(String tweak, String in, String format) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_US_PHONE_ENC, tweak);
            return sessionKey.decryptUSPhone(in, format);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Encrypt a SSN in format preserving form, the encrypted value is also a legitimate SSN
     * @param tweak The tweak used for the encryption
     * @param in SSN to encrypt
     * @param format The output format, e.g. "###-##-####"
     * @return The encrypted value
     * @throws SecurityException In case of encryption error
     */
    public String encryptSSN(String tweak, String in, String format) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SSN_ENC, tweak);
            return sessionKey.encryptSSN(in, format);
        }
        finally { sessionKey.destroy(); }
    }

    /**
     * Decrypt an encrypted SSN, the encrypted value is also SSN
     * @param tweak The tweak used for the decryption
     * @param in The encrypted value
     * @param format The output format, e.g. "###-##-####"
     * @return Original plain SSN
     * @throws SecurityException In case of decryption error
     */
    public String decryptSSN(String tweak, String in, String format) throws SecurityException
    {
        SDESessionKey sessionKey = null;
        try
        {
            sessionKey = generateSessionKey(PURPOSE_SSN_ENC, tweak);
            return sessionKey.decryptSSN(in, format);
        }
        finally { sessionKey.destroy(); }
    }

}
