/*
 *                    GNU LESSER GENERAL PUBLIC LICENSE
 *                        Version 3, 29 June 2007
 *
 *  Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *
 *   This version of the GNU Lesser General Public License incorporates
 * the terms and conditions of version 3 of the GNU General Public
 * License, supplemented by the additional permissions listed below.
 *
 *   0. Additional Definitions.
 *
 *   As used herein, "this License" refers to version 3 of the GNU Lesser
 * General Public License, and the "GNU GPL" refers to version 3 of the GNU
 * General Public License.
 *
 *   "The Library" refers to a covered work governed by this License,
 * other than an Application or a Combined Work as defined below.
 *
 *   An "Application" is any work that makes use of an interface provided
 * by the Library, but which is not otherwise based on the Library.
 * Defining a subclass of a class defined by the Library is deemed a mode
 * of using an interface provided by the Library.
 *
 *   A "Combined Work" is a work produced by combining or linking an
 * Application with the Library.  The particular version of the Library
 * with which the Combined Work was made is also called the "Linked
 * Version".
 *
 *   The "Minimal Corresponding Source" for a Combined Work means the
 * Corresponding Source for the Combined Work, excluding any source code
 * for portions of the Combined Work that, considered in isolation, are
 * based on the Application, and not on the Linked Version.
 *
 *   The "Corresponding Application Code" for a Combined Work means the
 * object code and/or source code for the Application, including any data
 * and utility programs needed for reproducing the Combined Work from the
 * Application, but excluding the System Libraries of the Combined Work.
 *
 *   1. Exception to Section 3 of the GNU GPL.
 *
 *   You may convey a covered work under sections 3 and 4 of this License
 * without being bound by section 3 of the GNU GPL.
 *
 *   2. Conveying Modified Versions.
 *
 *   If you modify a copy of the Library, and, in your modifications, a
 * facility refers to a function or data to be supplied by an Application
 * that uses the facility (other than as an argument passed when the
 * facility is invoked), then you may convey a copy of the modified
 * version:
 *
 *    a) under this License, provided that you make a good faith effort to
 *    ensure that, in the event an Application does not supply the
 *    function or data, the facility still operates, and performs
 *    whatever part of its purpose remains meaningful, or
 *
 *    b) under the GNU GPL, with none of the additional permissions of
 *    this License applicable to that copy.
 *
 *   3. Object Code Incorporating Material from Library Header Files.
 *
 *   The object code form of an Application may incorporate material from
 * a header file that is part of the Library.  You may convey such object
 * code under terms of your choice, provided that, if the incorporated
 * material is not limited to numerical parameters, data structure
 * layouts and accessors, or small macros, inline functions and templates
 * (ten or fewer lines in length), you do both of the following:
 *
 *    a) Give prominent notice with each copy of the object code that the
 *    Library is used in it and that the Library and its use are
 *    covered by this License.
 *
 *    b) Accompany the object code with a copy of the GNU GPL and this license
 *    document.
 *
 *   4. Combined Works.
 *
 *   You may convey a Combined Work under terms of your choice that,
 * taken together, effectively do not restrict modification of the
 * portions of the Library contained in the Combined Work and reverse
 * engineering for debugging such modifications, if you also do each of
 * the following:
 *
 *    a) Give prominent notice with each copy of the Combined Work that
 *    the Library is used in it and that the Library and its use are
 *    covered by this License.
 *
 *    b) Accompany the Combined Work with a copy of the GNU GPL and this license
 *    document.
 *
 *    c) For a Combined Work that displays copyright notices during
 *    execution, include the copyright notice for the Library among
 *    these notices, as well as a reference directing the user to the
 *    copies of the GNU GPL and this license document.
 *
 *    d) Do one of the following:
 *
 *        0) Convey the Minimal Corresponding Source under the terms of this
 *        License, and the Corresponding Application Code in a form
 *        suitable for, and under terms that permit, the user to
 *        recombine or relink the Application with a modified version of
 *        the Linked Version to produce a modified Combined Work, in the
 *        manner specified by section 6 of the GNU GPL for conveying
 *        Corresponding Source.
 *
 *        1) Use a suitable shared library mechanism for linking with the
 *        Library.  A suitable mechanism is one that (a) uses at run time
 *        a copy of the Library already present on the user's computer
 *        system, and (b) will operate properly with a modified version
 *        of the Library that is interface-compatible with the Linked
 *        Version.
 *
 *    e) Provide Installation Information, but only if you would otherwise
 *    be required to provide such information under section 6 of the
 *    GNU GPL, and only to the extent that such information is
 *    necessary to install and execute a modified version of the
 *    Combined Work produced by recombining or relinking the
 *    Application with a modified version of the Linked Version. (If
 *    you use option 4d0, the Installation Information must accompany
 *    the Minimal Corresponding Source and Corresponding Application
 *    Code. If you use option 4d1, you must provide the Installation
 *    Information in the manner specified by section 6 of the GNU GPL
 *    for conveying Corresponding Source.)
 *
 *   5. Combined Libraries.
 *
 *   You may place library facilities that are a work based on the
 * Library side by side in a single library together with other library
 * facilities that are not Applications and are not covered by this
 * License, and convey such a combined library under terms of your
 * choice, if you do both of the following:
 *
 *    a) Accompany the combined library with a copy of the same work based
 *    on the Library, uncombined with any other library facilities,
 *    conveyed under the terms of this License.
 *
 *    b) Give prominent notice with the combined library that part of it
 *    is a work based on the Library, and explaining where to find the
 *    accompanying uncombined form of the same work.
 *
 *   6. Revised Versions of the GNU Lesser General Public License.
 *
 *   The Free Software Foundation may publish revised and/or new versions
 * of the GNU Lesser General Public License from time to time. Such new
 * versions will be similar in spirit to the present version, but may
 * differ in detail to address new problems or concerns.
 *
 *   Each version is given a distinguishing version number. If the
 * Library as you received it specifies that a certain numbered version
 * of the GNU Lesser General Public License "or any later version"
 * applies to it, you have the option of following the terms and
 * conditions either of that published version or of any later version
 * published by the Free Software Foundation. If the Library as you
 * received it does not specify a version number of the GNU Lesser
 * General Public License, you may choose any version of the GNU Lesser
 * General Public License ever published by the Free Software Foundation.
 *
 *   If the Library as you received it specifies that a proxy can decide
 * whether future versions of the GNU Lesser General Public License shall
 * apply, that proxy's public statement of acceptance of any version is
 * permanent authorization for you to choose that version for the
 * Library.
 */
package com.github.szgabsz91.morpher.languagehandlers.hunmorph.impl.markov;

import com.github.szgabsz91.morpher.core.model.AffixType;
import com.github.szgabsz91.morpher.languagehandlers.api.model.AffixTypeChain;
import com.github.szgabsz91.morpher.languagehandlers.api.model.ProbabilisticAffixType;
import com.github.szgabsz91.morpher.languagehandlers.hunmorph.converters.FullMarkovModelConverter;
import com.github.szgabsz91.morpher.languagehandlers.hunmorph.protocolbuffers.MarkovModelMessage;
import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.IntStream;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

/**
 * {@link IMarkovModel} implementation that stores every single transition route.
 *
 * @author szgabsz91
 */
public class FullMarkovModel implements IMarkovModel {

    private Map<AffixType, Set<Node>> affixTypeNodeMap;

    /**
     * Default constructor.
     */
    public FullMarkovModel() {
        this.affixTypeNodeMap = new HashMap<>();
        this.affixTypeNodeMap.put(IMarkovModel.START, Collections.singleton(Node.start()));
    }

    /**
     * Returns a set of nodes associated with the given affix type, or null if the affix type is not contained
     * by this Markov model.
     *
     * @param affixType the affix type
     * @return the set of nodes or null
     */
    Set<Node> getNodes(final AffixType affixType) {
        return this.affixTypeNodeMap.getOrDefault(affixType, Collections.emptySet());
    }

    Node getStartNode() {
        return this.getNodes(IMarkovModel.START).iterator().next();
    }

    private void addNode(final AffixType affixType, final Node node) {
        final Set<Node> nodes = this.affixTypeNodeMap.computeIfAbsent(affixType, k -> new HashSet<>());
        nodes.add(node);
    }

    /**
     * Adds the given list of affix type to the Markov model.
     * @param affixTypes the list of affix types
     */
    @Override
    public void add(final List<AffixType> affixTypes) {
        if (affixTypes.isEmpty()) {
            return;
        }

        Node currentNode = this.getStartNode();
        currentNode.incrementRelativeFrequency();

        for (final AffixType affixType : affixTypes) {
            final Node previousNode = currentNode;
            currentNode = currentNode.getChild(affixType)
                    .orElseGet(() -> {
                        final Node node = Node.create(affixType, previousNode);
                        this.addNode(affixType, node);
                        return node;
                    });
            currentNode.incrementRelativeFrequency();
        }

        final Node lastNode = currentNode;
        final Node endNode = currentNode.getChild(IMarkovModel.END)
                .orElseGet(() -> {
                    final Node node = Node.end(lastNode);
                    this.addNode(IMarkovModel.END, node);
                    return node;
                });
        endNode.incrementRelativeFrequency();
    }

    /**
     * Adds the given list of affix types to the Markov model, and sets the last relative frequency
     * to the provided value.
     *
     * @param affixTypes the list of affix types
     * @param relativeFrequency the last relative frequency
     */
    public void add(final List<AffixType> affixTypes, final long relativeFrequency) {
        Node currentNode = this.getStartNode();

        for (final AffixType affixType : affixTypes) {
            currentNode.incrementRelativeFrequencyBy(relativeFrequency);
            final Node previousNode = currentNode;
            currentNode = currentNode.getChild(affixType)
                    .orElseGet(() -> {
                        final Node node = Node.create(affixType, previousNode);
                        this.addNode(affixType, node);
                        return node;
                    });
        }

        currentNode.incrementRelativeFrequencyBy(relativeFrequency);

        final Node lastNode = currentNode;
        final Node endNode = currentNode.getChild(IMarkovModel.END)
                .orElse(Node.end(lastNode));
        this.addNode(IMarkovModel.END, endNode);
        endNode.incrementRelativeFrequencyBy(relativeFrequency);
    }

    /**
     * Returns if the affix type chain is valid or not.
     *
     * {@inheritDoc}
     */
    @Override
    public boolean isAffixTypeChainValid(final List<AffixType> affixTypeChain) {
        Node currentNode = this.getStartNode();

        for (final AffixType affixType : affixTypeChain) {
            final Optional<Node> optionalChild = currentNode.getChild(affixType);

            if (!optionalChild.isPresent()) {
                return false;
            }

            currentNode = optionalChild.get();
        }

        return true;
    }

    /**
     * Returns an ordered list of {@link AffixTypeChain} objects.
     *
     * {@inheritDoc}
     */
    @Override
    public List<AffixTypeChain> sortAffixTypes(
            final Set<AffixType> affixTypes,
            final Predicate<AffixType> posPredicate) {
        final Predicate<Node> extendedPosPredicate = node -> {
            final AffixType affixType = node.getAffixType();
            return affixTypes.contains(affixType) || posPredicate.test(affixType);
        };
        final Set<AffixTypeChain> affixTypeChainSet = new HashSet<>();
        sortAffixTypes(
                Collections.emptyList(),
                this.getStartNode(),
                affixTypes,
                extendedPosPredicate,
                affixTypeChainSet
        );
        return new ArrayList<>(affixTypeChainSet);
    }

    /**
     * Creates a new {@link AffixTypeChain} containing the given affix types.
     *
     * {@inheritDoc}
     */
    @Override
    public AffixTypeChain calculateProbabilities(final List<AffixType> affixTypes) {
        final List<AffixType> extendedAffixTypes = new ArrayList<>(affixTypes.size() + 1);
        extendedAffixTypes.addAll(affixTypes);
        extendedAffixTypes.add(END);

        return this.getMostProbablePosForAffixType(extendedAffixTypes.get(0))
                .map(pos -> {
                    final List<ProbabilisticAffixType> probabilisticAffixTypes = new LinkedList<>();
                    probabilisticAffixTypes.add(pos);

                    Node currentNode = this.getStartNode().getChild(pos.getAffixType()).get();
                    boolean routeFound = true;

                    for (final AffixType affixType : extendedAffixTypes) {
                        if (routeFound) {
                            final Optional<Node> optionalChild = currentNode.getChild(affixType);

                            if (optionalChild.isPresent()) {
                                final Node child = optionalChild.get();
                                final long relativeFrequency = child.getRelativeFrequency();
                                final long totalRelativeFrequency = currentNode.getRelativeFrequency();
                                final double probability = relativeFrequency / (double) totalRelativeFrequency;
                                final ProbabilisticAffixType probabilisticAffixType = ProbabilisticAffixType.of(
                                        affixType,
                                        probability
                                );
                                probabilisticAffixTypes.add(probabilisticAffixType);
                                currentNode = child;
                            }
                            else {
                                routeFound = false;
                                final ProbabilisticAffixType probabilisticAffixType = ProbabilisticAffixType.of(
                                        affixType,
                                        0.0
                                );
                                probabilisticAffixTypes.add(probabilisticAffixType);
                            }
                        }
                        else {
                            final ProbabilisticAffixType probabilisticAffixType = ProbabilisticAffixType.of(
                                    affixType,
                                    0.0
                            );
                            probabilisticAffixTypes.add(probabilisticAffixType);
                        }
                    }

                    final List<ProbabilisticAffixType> filteredProbabilisticAffixTypes = probabilisticAffixTypes
                            .stream()
                            .filter(probabilisticAffixType -> !probabilisticAffixType.getAffixType().equals(END))
                            .collect(toList());
                    final double probability = probabilisticAffixTypes
                            .stream()
                            .mapToDouble(ProbabilisticAffixType::getProbability)
                            .reduce(1.0, (x, y) -> x * y);
                    return AffixTypeChain.of(filteredProbabilisticAffixTypes, probability);
                })
                .orElseGet(() -> {
                    final List<ProbabilisticAffixType> probabilisticAffixTypes = affixTypes
                            .stream()
                            .map(affixType -> ProbabilisticAffixType.of(affixType, 0.0))
                            .collect(toList());
                    final double probability = probabilisticAffixTypes
                            .stream()
                            .mapToDouble(ProbabilisticAffixType::getProbability)
                            .reduce(1.0, (x, y) -> x * y);
                    return AffixTypeChain.of(probabilisticAffixTypes, probability);
                });
    }

    /**
     * Calculates the probability of the affix type chain given with its list of {@link AffixType}s.
     *
     * {@inheritDoc}
     */
    @Override
    public double calculateProbability(final List<AffixType> affixTypes) {
        final List<ProbabilisticAffixType> probabilisticAffixTypes = new LinkedList<>();
        final ProbabilisticAffixType pos = this.getProbabilityOfPOS(affixTypes.get(0));
        probabilisticAffixTypes.add(pos);

        Node currentNode = this.getStartNode().getChild(pos.getAffixType()).get();
        boolean routeFound = true;

        for (int i = 1; i < affixTypes.size(); i++) {
            final AffixType affixType = affixTypes.get(i);

            if (routeFound) {
                final Optional<Node> optionalChild = currentNode.getChild(affixType);

                if (optionalChild.isPresent()) {
                    final Node child = optionalChild.get();
                    final long relativeFrequency = child.getRelativeFrequency();
                    final long totalRelativeFrequency = currentNode.getChildren().values()
                            .stream()
                            .mapToLong(Node::getRelativeFrequency)
                            .sum();
                    final double probability = relativeFrequency / (double) totalRelativeFrequency;
                    final ProbabilisticAffixType probabilisticAffixType = ProbabilisticAffixType.of(
                            affixType,
                            probability
                    );
                    probabilisticAffixTypes.add(probabilisticAffixType);
                    currentNode = child;
                }
                else {
                    routeFound = false;
                    final ProbabilisticAffixType probabilisticAffixType = ProbabilisticAffixType.of(affixType, 0.0);
                    probabilisticAffixTypes.add(probabilisticAffixType);
                }
            }
            else {
                final ProbabilisticAffixType probabilisticAffixType = ProbabilisticAffixType.of(affixType, 0.0);
                probabilisticAffixTypes.add(probabilisticAffixType);
            }
        }

        if (!routeFound) {
            return 0.0;
        }

        return this.getProbabilityOf(probabilisticAffixTypes);
    }

    /**
     * Returns the affix type with its probability for the given part of speech tag.
     *
     * {@inheritDoc}
     */
    @Override
    public ProbabilisticAffixType getProbabilityOfPOS(final AffixType pos) {
        final Node startNode = this.getStartNode();

        final long totalRelativeFrequency = startNode.getChildren().values()
                .stream()
                .mapToLong(Node::getRelativeFrequency)
                .sum();

        return startNode.getChild(pos)
                .map(child -> {
                    final double probability = child.getRelativeFrequency() / (double) totalRelativeFrequency;
                    return ProbabilisticAffixType.of(pos, probability);
                })
                .orElse(ProbabilisticAffixType.of(pos, 0.0));
    }

    /**
     * Returns the list of candidates that can come after the given list of affix types.
     *
     * {@inheritDoc}
     */
    @Override
    public List<ProbabilisticAffixType> getCandidates(final List<AffixType> affixTypes) {
        Node currentNode = this.getStartNode();

        for (final AffixType affixType : affixTypes) {
            final Optional<Node> optionalNewNode = currentNode.getChild(affixType);

            if (!optionalNewNode.isPresent()) {
                return Collections.emptyList();
            }

            currentNode = optionalNewNode.get();
        }

        final long totalRelativeFrequencies = currentNode.getChildren().values()
                .stream()
                .mapToLong(Node::getRelativeFrequency)
                .sum();

        return currentNode.getChildren().values()
                .stream()
                .map(node -> {
                    final AffixType nextAffixType = node.getAffixType();
                    final long relativeFrequency = node.getRelativeFrequency();
                    final double probability = relativeFrequency / (double) totalRelativeFrequencies;
                    return ProbabilisticAffixType.of(nextAffixType, probability);
                })
                .collect(toList());
    }

    /**
     * Reverses this {@link FullMarkovModel}.
     * @return the reversed {@link FullMarkovModel}
     */
    @Override
    public IMarkovModel reverse() {
        final FullMarkovModel result = new FullMarkovModel();
        this.getRoutes().forEach((route, frequency) -> {
            final List<AffixType> reversedRoute = IntStream.range(0, route.size())
                    .map(index -> route.size() - index - 1)
                    .mapToObj(route::get)
                    .map(Node::getAffixType)
                    .collect(toList());
            result.add(reversedRoute, frequency);
        });
        return result;
    }

    /**
     * Converts this Markov model to a {@link MarkovModelMessage}.
     * @return the {@link MarkovModelMessage}
     */
    @Override
    public MarkovModelMessage toMessage() {
        final FullMarkovModelConverter converter = new FullMarkovModelConverter();
        return converter.convert(this);
    }

    /**
     * Loads the state from the given {@link MarkovModelMessage}.
     * @param message the {@link MarkovModelMessage}
     */
    @Override
    public void fromMessage(final MarkovModelMessage message) {
        this.affixTypeNodeMap.clear();
        this.affixTypeNodeMap.put(IMarkovModel.START, Collections.singleton(Node.start()));
        final FullMarkovModelConverter converter = new FullMarkovModelConverter();
        final FullMarkovModel fullMarkovModel = converter.convertBack(message);
        this.affixTypeNodeMap = fullMarkovModel.affixTypeNodeMap;
    }

    /**
     * Loads the state from the given message.
     * @param message the message
     * @throws InvalidProtocolBufferException if the given message is not a {@link MarkovModelMessage}
     */
    @Override
    public void fromMessage(final Any message) throws InvalidProtocolBufferException {
        if (!message.is(MarkovModelMessage.class)) {
            throw new InvalidProtocolBufferException("The provided message is not a MarkovModelMessage: " + message);
        }

        final MarkovModelMessage markovModelMessage = message.unpack(MarkovModelMessage.class);
        this.fromMessage(markovModelMessage);
    }

    /**
     * Returns all the routes in this Markov model, with the relative frequency of the last node.
     * @return all the routes associated with the relative frequency of the last node
     */
    public Map<List<Node>, Long> getRoutes() {
        return this.getNodes(IMarkovModel.END)
                .stream()
                .map(endNode -> {
                    final List<Node> route = new LinkedList<>();
                    Node currentNode = endNode.getParent().get();
                    while (!IMarkovModel.START.equals(currentNode.getAffixType())) {
                        route.add(0, currentNode);
                        currentNode = currentNode.getParent().get();
                    }
                    return route;
                })
                .collect(toMap(Function.identity(), route -> {
                    return route.get(route.size() - 1).getChild(IMarkovModel.END).get().getRelativeFrequency();
                }));
    }

    private void sortAffixTypes(
            final List<Node> previousNodes,
            final Node currentNode,
            final Set<AffixType> requiredAffixTypes,
            final Predicate<Node> posPredicate,
            final Set<AffixTypeChain> sink) {
        final List<Node> extendedNodes = new ArrayList<>(previousNodes.size() + 1);
        extendedNodes.addAll(previousNodes);
        extendedNodes.add(currentNode);
        final Set<AffixType> extendedAffixTypes = extendedNodes
                .stream()
                .map(Node::getAffixType)
                .collect(toSet());

        if (extendedAffixTypes.containsAll(requiredAffixTypes)) {
            final List<ProbabilisticAffixType> probabilisticAffixTypes = extendedNodes
                    .stream()
                    .filter(posPredicate)
                    .map(node -> {
                        final AffixType affixType = node.getAffixType();
                        final long relativeFrequency = node.getRelativeFrequency();
                        final long totalRelativeFrequencies = node.getParent().get().getChildren().values()
                                .stream()
                                .mapToLong(Node::getRelativeFrequency)
                                .sum();
                        final double probability = relativeFrequency / (double) totalRelativeFrequencies;
                        return ProbabilisticAffixType.of(affixType, probability);
                    })
                    .collect(toList());
            final double probability = probabilisticAffixTypes
                    .stream()
                    .mapToDouble(ProbabilisticAffixType::getProbability)
                    .reduce(1.0, (x, y) -> x * y);
            final AffixTypeChain affixTypeChain = AffixTypeChain.of(probabilisticAffixTypes, probability);
            sink.add(affixTypeChain);
            return;
        }

        currentNode.getChildren().values()
                .stream()
                .filter(posPredicate)
                .distinct()
                .forEach(node -> {
                    sortAffixTypes(
                            extendedNodes,
                            node,
                            requiredAffixTypes,
                            posPredicate,
                            sink
                    );
                });
    }

    private Optional<ProbabilisticAffixType> getMostProbablePosForAffixType(final AffixType affixType) {
        final Node startNode = this.getStartNode();
        final double totalRelativeFrequency = startNode.getRelativeFrequency();

        return startNode.getChildren().values()
                .stream()
                .filter(node -> node.getChild(affixType).isPresent())
                .findFirst()
                .map(node -> {
                    final long relativeFrequency = node.getRelativeFrequency();
                    final double probability = relativeFrequency / totalRelativeFrequency;
                    return ProbabilisticAffixType.of(node.getAffixType(), probability);
                });
    }

    private double getProbabilityOf(final List<ProbabilisticAffixType> probabilisticAffixTypes) {
        final List<AffixType> nominatorAffixTypes = probabilisticAffixTypes
                .stream()
                .map(ProbabilisticAffixType::getAffixType)
                .collect(toList());
        final List<AffixType> denominatorAffixTypes = new ArrayList<>(nominatorAffixTypes);
        denominatorAffixTypes.remove(denominatorAffixTypes.size() - 1);
        final double nominator = this.getRelativeFrequencyOfRoutesContaining(nominatorAffixTypes);
        final double denominator = this.getRelativeFrequencyOfRoutesContaining(denominatorAffixTypes);
        return nominator / denominator;
    }

    private long getRelativeFrequencyOfRoutesContaining(final List<AffixType> affixTypes) {
        Node currentNode = this.getStartNode();

        for (final AffixType affixType : affixTypes) {
            final Optional<Node> optionalChild = currentNode.getChild(affixType);
            if (!optionalChild.isPresent()) {
                return 0L;
            }
            currentNode = optionalChild.get();
        }

        return currentNode.getRelativeFrequency();
    }

    /**
     * Returns the hierarchical string representation of this Markov model.
     *
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        printNode(sb, this.getStartNode(), 0);
        return sb.toString();
    }

    private static void printNode(final StringBuilder sb, final Node node, final int level) {
        for (int i = 0; i < level; i++) {
            sb.append("  ");
        }

        sb.append(node.getAffixType());
        sb.append(':');
        sb.append(node.getRelativeFrequency());
        sb.append(System.getProperty("line.separator"));

        node.getChildren().values().forEach(child -> printNode(sb, child, level + 1));
    }

    /**
     * Node class for the {@link FullMarkovModel}.
     *
     * @author szgabsz91
     */
    public static class Node implements Comparable<Node> {

        private final AffixType affixType;
        private long relativeFrequency;
        private final Node parent;
        private final Map<AffixType, Node> children;

        /**
         * Constructor that sets the affix type, relative frequency and the optional parent.
         * @param affixType the affix type
         * @param relativeFrequency the relative frequency
         * @param parent the optional parent
         */
        public Node(final AffixType affixType, final long relativeFrequency, final Node parent) {
            this.affixType = affixType;
            this.relativeFrequency = relativeFrequency;
            this.parent = parent;
            this.children = new HashMap<>();

            if (parent != null) {
                parent.getChildren().put(affixType, this);
            }
        }

        /**
         * Returns a new start node with a relative frequency of -1.
         * @return the new start node
         */
        public static Node start() {
            return new Node(IMarkovModel.START, 0L, null);
        }

        /**
         * Returns a new end node and sets its parent to the given node, as well as its relative frequency to one.
         * @param parent the parent of the new end node
         * @return the new end node
         */
        public static Node end(final Node parent) {
            return new Node(IMarkovModel.END, 0L, parent);
        }

        /**
         * Creates a new node with the given affix type and parent, setting its relative frequency to zero.
         * @param affixType the affix type
         * @param parent the parent
         * @return the new node
         */
        public static Node create(final AffixType affixType, final Node parent) {
            return new Node(affixType, 0L, parent);
        }

        /**
         * Returns the affix type.
         * @return the affix type
         */
        public AffixType getAffixType() {
            return affixType;
        }

        /**
         * Returns the relative frequency.
         * @return the relative frequency
         */
        public long getRelativeFrequency() {
            return relativeFrequency;
        }

        /**
         * Increments the relative frequency by 1.
         */
        public void incrementRelativeFrequency() {
            this.relativeFrequency++;
        }

        /**
         * Increments the relative frequency by the given value.
         * @param relativeFrequencyIncrement the relative frequency to increment by
         */
        public void incrementRelativeFrequencyBy(final long relativeFrequencyIncrement) {
            this.relativeFrequency += relativeFrequencyIncrement;
        }

        /**
         * Returns the optional parent node.
         * @return the optional parent node
         */
        public Optional<Node> getParent() {
            return Optional.ofNullable(parent);
        }

        /**
         * Returns the map of child nodes.
         * @return the map of child nodes
         */
        public Map<AffixType, Node> getChildren() {
            return children;
        }

        /**
         * Returns the optional child node for the given affix type.
         * @param affixType the affix type
         * @return the optional child node
         */
        public Optional<Node> getChild(final AffixType affixType) {
            return Optional.ofNullable(this.children.get(affixType));
        }

        /**
         * Compares the objects by their probabilities.
         * @param other the other node
         * @return -1 if the given object has higher probability, 0 is they are equal, +1 otherwise
         */
        @Override
        public int compareTo(final Node other) {
            return Long.compare(other.relativeFrequency, this.relativeFrequency);
        }

        /**
         * Returns if this node equals the given other object.
         *
         * {@inheritDoc}
         */
        @Override
        public boolean equals(final Object other) {
            if (this == other) {
                return true;
            }
            if (other == null || getClass() != other.getClass()) {
                return false;
            }
            final Node node = (Node) other;
            return relativeFrequency == node.relativeFrequency &&
                    affixType.equals(node.affixType) &&
                    (parent != null ? parent.equals(node.parent) : node.parent == null);
        }

        /**
         * Returns the hash code of this node.
         *
         * {@inheritDoc}
         */
        @Override
        public int hashCode() {
            int result = affixType.hashCode();
            result = 31 * result + (int) (relativeFrequency ^ (relativeFrequency >>> 32));
            result = 31 * result + (parent != null ? parent.hashCode() : 0);
            return result;
        }

        /**
         * Returns the string representation of this node.
         *
         * {@inheritDoc}
         */
        @Override
        public String toString() {
            return "Node[" +
                    "affixType=" + affixType +
                    ", relativeFrequency=" + relativeFrequency +
                    ", parent=" + parent +
                    ']';
        }

    }

}
