/*
 *                    GNU LESSER GENERAL PUBLIC LICENSE
 *                        Version 3, 29 June 2007
 *
 *  Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *
 *   This version of the GNU Lesser General Public License incorporates
 * the terms and conditions of version 3 of the GNU General Public
 * License, supplemented by the additional permissions listed below.
 *
 *   0. Additional Definitions.
 *
 *   As used herein, "this License" refers to version 3 of the GNU Lesser
 * General Public License, and the "GNU GPL" refers to version 3 of the GNU
 * General Public License.
 *
 *   "The Library" refers to a covered work governed by this License,
 * other than an Application or a Combined Work as defined below.
 *
 *   An "Application" is any work that makes use of an interface provided
 * by the Library, but which is not otherwise based on the Library.
 * Defining a subclass of a class defined by the Library is deemed a mode
 * of using an interface provided by the Library.
 *
 *   A "Combined Work" is a work produced by combining or linking an
 * Application with the Library.  The particular version of the Library
 * with which the Combined Work was made is also called the "Linked
 * Version".
 *
 *   The "Minimal Corresponding Source" for a Combined Work means the
 * Corresponding Source for the Combined Work, excluding any source code
 * for portions of the Combined Work that, considered in isolation, are
 * based on the Application, and not on the Linked Version.
 *
 *   The "Corresponding Application Code" for a Combined Work means the
 * object code and/or source code for the Application, including any data
 * and utility programs needed for reproducing the Combined Work from the
 * Application, but excluding the System Libraries of the Combined Work.
 *
 *   1. Exception to Section 3 of the GNU GPL.
 *
 *   You may convey a covered work under sections 3 and 4 of this License
 * without being bound by section 3 of the GNU GPL.
 *
 *   2. Conveying Modified Versions.
 *
 *   If you modify a copy of the Library, and, in your modifications, a
 * facility refers to a function or data to be supplied by an Application
 * that uses the facility (other than as an argument passed when the
 * facility is invoked), then you may convey a copy of the modified
 * version:
 *
 *    a) under this License, provided that you make a good faith effort to
 *    ensure that, in the event an Application does not supply the
 *    function or data, the facility still operates, and performs
 *    whatever part of its purpose remains meaningful, or
 *
 *    b) under the GNU GPL, with none of the additional permissions of
 *    this License applicable to that copy.
 *
 *   3. Object Code Incorporating Material from Library Header Files.
 *
 *   The object code form of an Application may incorporate material from
 * a header file that is part of the Library.  You may convey such object
 * code under terms of your choice, provided that, if the incorporated
 * material is not limited to numerical parameters, data structure
 * layouts and accessors, or small macros, inline functions and templates
 * (ten or fewer lines in length), you do both of the following:
 *
 *    a) Give prominent notice with each copy of the object code that the
 *    Library is used in it and that the Library and its use are
 *    covered by this License.
 *
 *    b) Accompany the object code with a copy of the GNU GPL and this license
 *    document.
 *
 *   4. Combined Works.
 *
 *   You may convey a Combined Work under terms of your choice that,
 * taken together, effectively do not restrict modification of the
 * portions of the Library contained in the Combined Work and reverse
 * engineering for debugging such modifications, if you also do each of
 * the following:
 *
 *    a) Give prominent notice with each copy of the Combined Work that
 *    the Library is used in it and that the Library and its use are
 *    covered by this License.
 *
 *    b) Accompany the Combined Work with a copy of the GNU GPL and this license
 *    document.
 *
 *    c) For a Combined Work that displays copyright notices during
 *    execution, include the copyright notice for the Library among
 *    these notices, as well as a reference directing the user to the
 *    copies of the GNU GPL and this license document.
 *
 *    d) Do one of the following:
 *
 *        0) Convey the Minimal Corresponding Source under the terms of this
 *        License, and the Corresponding Application Code in a form
 *        suitable for, and under terms that permit, the user to
 *        recombine or relink the Application with a modified version of
 *        the Linked Version to produce a modified Combined Work, in the
 *        manner specified by section 6 of the GNU GPL for conveying
 *        Corresponding Source.
 *
 *        1) Use a suitable shared library mechanism for linking with the
 *        Library.  A suitable mechanism is one that (a) uses at run time
 *        a copy of the Library already present on the user's computer
 *        system, and (b) will operate properly with a modified version
 *        of the Library that is interface-compatible with the Linked
 *        Version.
 *
 *    e) Provide Installation Information, but only if you would otherwise
 *    be required to provide such information under section 6 of the
 *    GNU GPL, and only to the extent that such information is
 *    necessary to install and execute a modified version of the
 *    Combined Work produced by recombining or relinking the
 *    Application with a modified version of the Linked Version. (If
 *    you use option 4d0, the Installation Information must accompany
 *    the Minimal Corresponding Source and Corresponding Application
 *    Code. If you use option 4d1, you must provide the Installation
 *    Information in the manner specified by section 6 of the GNU GPL
 *    for conveying Corresponding Source.)
 *
 *   5. Combined Libraries.
 *
 *   You may place library facilities that are a work based on the
 * Library side by side in a single library together with other library
 * facilities that are not Applications and are not covered by this
 * License, and convey such a combined library under terms of your
 * choice, if you do both of the following:
 *
 *    a) Accompany the combined library with a copy of the same work based
 *    on the Library, uncombined with any other library facilities,
 *    conveyed under the terms of this License.
 *
 *    b) Give prominent notice with the combined library that part of it
 *    is a work based on the Library, and explaining where to find the
 *    accompanying uncombined form of the same work.
 *
 *   6. Revised Versions of the GNU Lesser General Public License.
 *
 *   The Free Software Foundation may publish revised and/or new versions
 * of the GNU Lesser General Public License from time to time. Such new
 * versions will be similar in spirit to the present version, but may
 * differ in detail to address new problems or concerns.
 *
 *   Each version is given a distinguishing version number. If the
 * Library as you received it specifies that a certain numbered version
 * of the GNU Lesser General Public License "or any later version"
 * applies to it, you have the option of following the terms and
 * conditions either of that published version or of any later version
 * published by the Free Software Foundation. If the Library as you
 * received it does not specify a version number of the GNU Lesser
 * General Public License, you may choose any version of the GNU Lesser
 * General Public License ever published by the Free Software Foundation.
 *
 *   If the Library as you received it specifies that a proxy can decide
 * whether future versions of the GNU Lesser General Public License shall
 * apply, that proxy's public statement of acceptance of any version is
 * permanent authorization for you to choose that version for the
 * Library.
 */
package com.github.szgabsz91.morpher.languagehandlers.hunmorph.impl;

import com.github.szgabsz91.morpher.languagehandlers.hunmorph.impl.model.HunmorphResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import static java.util.stream.Collectors.toList;

/**
 * Processes words using Hunmorph-Ocamorph.
 *
 * @author szgabsz91
 */
public class HunmorphWordProcessor implements AutoCloseable {

    private static final Logger LOGGER = LoggerFactory.getLogger(HunmorphWordProcessor.class);
    private static final Path FILE_HUNMORPH_BIN = Paths.get(
            "morpher-language-handlers/morpher-language-handler-hunmorph/src/main/resources/" +
            HunmorphWordProcessor.class.getPackageName().replace('.', '/') +
            "/hunmorph.bin"
    );
    private static final String OCAMORPH_COMMAND;

    static {
        OCAMORPH_COMMAND = getOcamorphCommand();
    }

    private Path temporaryDirectoryPath;
    private Path inputPath;
    private Path outputPath;
    private Path binPath;

    /**
     * Default constructor that creates the hunmorph.bin file as a temporary file automatically.
     */
    public HunmorphWordProcessor() {
        this(temporaryDirectoryPath -> {
            try {
                return Files.createTempFile(temporaryDirectoryPath, "hunmorph.bin", null);
            }
            catch (final IOException e) {
                throw new IllegalStateException("Cannot create hunmorph.bin temporary file", e);
            }
        });
    }

    /**
     * Constructor that retrieves the hunmorph.bin file from the provided supplier.
     * @param binFileSupplier the supplier that returns the hunmorph.bin file
     */
    public HunmorphWordProcessor(final Function<Path, Path> binFileSupplier) {
        this.createTemporaryDirectory();
        this.binPath = binFileSupplier.apply(this.temporaryDirectoryPath);
        createTemporaryFiles();
    }

    /**
     * Deletes the temporary files and folders.
     */
    @Override
    public void close() {
        try {
            Files.walkFileTree(temporaryDirectoryPath, new DeleteTemporaryDirectoryFileVisitor());
        }
        catch (final IOException e) {
            LOGGER.error("I/O exception while recursively deleting temporary directory", e);
        }
    }

    /**
     * Processes the given grammatical form and returns the resulting optional {@link HunmorphResult} instance.
     * @param grammaticalForm the grammatical form
     * @return the resulting optional {@link HunmorphResult} instance
     */
    public Optional<HunmorphResult> process(final String grammaticalForm) {
        return this.process(grammaticalForm, false);
    }

    /**
     * Processes the given grammatical form and returns the resulting optional {@link HunmorphResult} instance.
     * @param grammaticalForm the grammatical form
     * @param guess flag that indicates if unknown words should be analyzed or not
     * @return the resulting optional {@link HunmorphResult} instance
     */
    public Optional<HunmorphResult> process(final String grammaticalForm, final boolean guess) {
        LOGGER.debug("Processing {} with hunmorph", grammaticalForm);

        // Writing input to a temporary file
        try (final BufferedWriter inputWriter = Files.newBufferedWriter(inputPath, Charset.forName("ISO-8859-2"))) {
            inputWriter.write(grammaticalForm);
            inputWriter.flush();
        }
        catch (final IOException e) {
            throw new IllegalStateException("I/O exception while writing hunmorph input file", e);
        }

        // Invoke ocamorph, log its output and error code
        final List<String> command = new ArrayList<>(List.of(
                OCAMORPH_COMMAND,
                "--in", String.format("%s", inputPath.toAbsolutePath()),
                "--out", String.format("%s", outputPath.toAbsolutePath()),
                "--bin", String.format("%s", binPath.toAbsolutePath())
        ));
        if (guess) {
            command.addAll(List.of("--guess", "Global"));
        }
        final ProcessBuilder processBuilder = new ProcessBuilder(command);
        final int exitCode = invokeProcess(processBuilder);
        if (exitCode != 0) {
            throw new IllegalStateException("Exit code of ocamorph: " + exitCode);
        }

        // Process output of ocamorph
        try (final BufferedReader outputReader = Files.newBufferedReader(outputPath, Charset.forName("ISO-8859-2"))) {
            final List<String> outputLines = outputReader
                    .lines()
                    .collect(toList());
            outputLines.remove(0);
            if (outputLines.get(0).contains("UNKNOWN")) {
                return Optional.empty();
            }
            LOGGER.trace("Hunmorph result: {}", outputLines);
            return Optional.of(new HunmorphResult(grammaticalForm, outputLines));
        }
        catch (final IOException e) {
            throw new IllegalStateException("I/O exception while reading hunmorph output file", e);
        }
    }

    /**
     * Invokes the ocamorph process.
     *
     * <p>Reads the process output and error streams and logs them. Then, returns the exit code of the process.</p>
     *
     * @param processBuilder the process builder
     * @return the exit code of ocamorph
     */
    private static int invokeProcess(final ProcessBuilder processBuilder) {
        try {
            processBuilder.redirectErrorStream(true);
            final Process process = processBuilder.start();
            final int exitCode = process.waitFor();
            try (final InputStream inputStream = process.getInputStream();
                    final InputStreamReader reader = new InputStreamReader(inputStream, StandardCharsets.UTF_8);
                    final BufferedReader bufferedReader = new BufferedReader(reader)) {
                bufferedReader
                        .lines()
                        .forEach(outputLine -> LOGGER.trace("Output from ocamorph: {}", outputLine));
            }
            return exitCode;
        }
        catch (final IOException | InterruptedException e) {
            LOGGER.error("Exception while invoking ocamorph", e);
            return -1;
        }
    }

    /**
     * Creates a temporary directory for hunmorph input and output files.
     */
    private void createTemporaryDirectory() {
        try {
            temporaryDirectoryPath = Files.createTempDirectory("hunmorph-files");

            LOGGER.trace(
                    "Temporary directory created: {}",
                    temporaryDirectoryPath.toAbsolutePath().toString()
            );
        }
        catch (final IOException e) {
            throw new IllegalStateException("Cannot create temporary folder", e);
        }
    }

    /**
     * Creates temporary files for hunmorph input and output.
     */
    private void createTemporaryFiles() {
        try (final InputStream inputStream = HunmorphWordProcessor.class.getResourceAsStream("hunmorph.bin")) {
            inputPath = Files.createTempFile(temporaryDirectoryPath, "input", null);
            outputPath = Files.createTempFile(temporaryDirectoryPath, "output", null);
            Files.delete(binPath);
            if (inputStream != null) {
                Files.copy(inputStream, binPath);
            }
            else {
                Files.copy(FILE_HUNMORPH_BIN, binPath);
            }

            LOGGER.trace(
                    "Temporary files created: {}, {} and {}",
                    inputPath.toAbsolutePath().toString(),
                    outputPath.toAbsolutePath().toString(),
                    binPath.toAbsolutePath().toString()
            );
        }
        catch (final IOException e) {
            throw new IllegalStateException("Cannot create temporary hunmorph files", e);
        }
    }

    /**
     * Returns the fully qualified command path of ocamorph based on the operating system.
     * @return the ocamorph command
     */
    private static String getOcamorphCommand() {
        final String osName = System.getProperty("os.name").toLowerCase();

        if (osName.contains("mac")) {
            return "/usr/local/bin/ocamorph";
        }

        return "/usr/bin/ocamorph";
    }

    /**
     * File visitor that recursively deletes the temporary directory.
     */
    private static class DeleteTemporaryDirectoryFileVisitor extends SimpleFileVisitor<Path> {

        /**
         * Deletes the given file and continues the traversal.
         *
         * {@inheritDoc}
         */
        @Override
        public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException {
            Files.delete(file);
            return FileVisitResult.CONTINUE;
        }

        /**
         * Deletes the given folder and continues the traversal.
         *
         * {@inheritDoc}
         */
        @Override
        public FileVisitResult postVisitDirectory(final Path folder, final IOException exception) throws IOException {
            Files.delete(folder);
            return FileVisitResult.CONTINUE;
        }

    }

}
