/*
 *                    GNU LESSER GENERAL PUBLIC LICENSE
 *                        Version 3, 29 June 2007
 *
 *  Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *
 *   This version of the GNU Lesser General Public License incorporates
 * the terms and conditions of version 3 of the GNU General Public
 * License, supplemented by the additional permissions listed below.
 *
 *   0. Additional Definitions.
 *
 *   As used herein, "this License" refers to version 3 of the GNU Lesser
 * General Public License, and the "GNU GPL" refers to version 3 of the GNU
 * General Public License.
 *
 *   "The Library" refers to a covered work governed by this License,
 * other than an Application or a Combined Work as defined below.
 *
 *   An "Application" is any work that makes use of an interface provided
 * by the Library, but which is not otherwise based on the Library.
 * Defining a subclass of a class defined by the Library is deemed a mode
 * of using an interface provided by the Library.
 *
 *   A "Combined Work" is a work produced by combining or linking an
 * Application with the Library.  The particular version of the Library
 * with which the Combined Work was made is also called the "Linked
 * Version".
 *
 *   The "Minimal Corresponding Source" for a Combined Work means the
 * Corresponding Source for the Combined Work, excluding any source code
 * for portions of the Combined Work that, considered in isolation, are
 * based on the Application, and not on the Linked Version.
 *
 *   The "Corresponding Application Code" for a Combined Work means the
 * object code and/or source code for the Application, including any data
 * and utility programs needed for reproducing the Combined Work from the
 * Application, but excluding the System Libraries of the Combined Work.
 *
 *   1. Exception to Section 3 of the GNU GPL.
 *
 *   You may convey a covered work under sections 3 and 4 of this License
 * without being bound by section 3 of the GNU GPL.
 *
 *   2. Conveying Modified Versions.
 *
 *   If you modify a copy of the Library, and, in your modifications, a
 * facility refers to a function or data to be supplied by an Application
 * that uses the facility (other than as an argument passed when the
 * facility is invoked), then you may convey a copy of the modified
 * version:
 *
 *    a) under this License, provided that you make a good faith effort to
 *    ensure that, in the event an Application does not supply the
 *    function or data, the facility still operates, and performs
 *    whatever part of its purpose remains meaningful, or
 *
 *    b) under the GNU GPL, with none of the additional permissions of
 *    this License applicable to that copy.
 *
 *   3. Object Code Incorporating Material from Library Header Files.
 *
 *   The object code form of an Application may incorporate material from
 * a header file that is part of the Library.  You may convey such object
 * code under terms of your choice, provided that, if the incorporated
 * material is not limited to numerical parameters, data structure
 * layouts and accessors, or small macros, inline functions and templates
 * (ten or fewer lines in length), you do both of the following:
 *
 *    a) Give prominent notice with each copy of the object code that the
 *    Library is used in it and that the Library and its use are
 *    covered by this License.
 *
 *    b) Accompany the object code with a copy of the GNU GPL and this license
 *    document.
 *
 *   4. Combined Works.
 *
 *   You may convey a Combined Work under terms of your choice that,
 * taken together, effectively do not restrict modification of the
 * portions of the Library contained in the Combined Work and reverse
 * engineering for debugging such modifications, if you also do each of
 * the following:
 *
 *    a) Give prominent notice with each copy of the Combined Work that
 *    the Library is used in it and that the Library and its use are
 *    covered by this License.
 *
 *    b) Accompany the Combined Work with a copy of the GNU GPL and this license
 *    document.
 *
 *    c) For a Combined Work that displays copyright notices during
 *    execution, include the copyright notice for the Library among
 *    these notices, as well as a reference directing the user to the
 *    copies of the GNU GPL and this license document.
 *
 *    d) Do one of the following:
 *
 *        0) Convey the Minimal Corresponding Source under the terms of this
 *        License, and the Corresponding Application Code in a form
 *        suitable for, and under terms that permit, the user to
 *        recombine or relink the Application with a modified version of
 *        the Linked Version to produce a modified Combined Work, in the
 *        manner specified by section 6 of the GNU GPL for conveying
 *        Corresponding Source.
 *
 *        1) Use a suitable shared library mechanism for linking with the
 *        Library.  A suitable mechanism is one that (a) uses at run time
 *        a copy of the Library already present on the user's computer
 *        system, and (b) will operate properly with a modified version
 *        of the Library that is interface-compatible with the Linked
 *        Version.
 *
 *    e) Provide Installation Information, but only if you would otherwise
 *    be required to provide such information under section 6 of the
 *    GNU GPL, and only to the extent that such information is
 *    necessary to install and execute a modified version of the
 *    Combined Work produced by recombining or relinking the
 *    Application with a modified version of the Linked Version. (If
 *    you use option 4d0, the Installation Information must accompany
 *    the Minimal Corresponding Source and Corresponding Application
 *    Code. If you use option 4d1, you must provide the Installation
 *    Information in the manner specified by section 6 of the GNU GPL
 *    for conveying Corresponding Source.)
 *
 *   5. Combined Libraries.
 *
 *   You may place library facilities that are a work based on the
 * Library side by side in a single library together with other library
 * facilities that are not Applications and are not covered by this
 * License, and convey such a combined library under terms of your
 * choice, if you do both of the following:
 *
 *    a) Accompany the combined library with a copy of the same work based
 *    on the Library, uncombined with any other library facilities,
 *    conveyed under the terms of this License.
 *
 *    b) Give prominent notice with the combined library that part of it
 *    is a work based on the Library, and explaining where to find the
 *    accompanying uncombined form of the same work.
 *
 *   6. Revised Versions of the GNU Lesser General Public License.
 *
 *   The Free Software Foundation may publish revised and/or new versions
 * of the GNU Lesser General Public License from time to time. Such new
 * versions will be similar in spirit to the present version, but may
 * differ in detail to address new problems or concerns.
 *
 *   Each version is given a distinguishing version number. If the
 * Library as you received it specifies that a certain numbered version
 * of the GNU Lesser General Public License "or any later version"
 * applies to it, you have the option of following the terms and
 * conditions either of that published version or of any later version
 * published by the Free Software Foundation. If the Library as you
 * received it does not specify a version number of the GNU Lesser
 * General Public License, you may choose any version of the GNU Lesser
 * General Public License ever published by the Free Software Foundation.
 *
 *   If the Library as you received it specifies that a proxy can decide
 * whether future versions of the GNU Lesser General Public License shall
 * apply, that proxy's public statement of acceptance of any version is
 * permanent authorization for you to choose that version for the
 * Library.
 */
package com.github.szgabsz91.morpher.languagehandlers.hunmorph.impl;

import com.github.szgabsz91.morpher.languagehandlers.api.ILanguageHandler;
import com.github.szgabsz91.morpher.languagehandlers.api.model.AffixTypeChain;
import com.github.szgabsz91.morpher.languagehandlers.api.model.AnnotationTokenizerResult;
import com.github.szgabsz91.morpher.languagehandlers.api.model.LanguageHandlerResponse;
import com.github.szgabsz91.morpher.languagehandlers.api.model.LemmaMap;
import com.github.szgabsz91.morpher.languagehandlers.api.model.ProbabilisticAffixType;
import com.github.szgabsz91.morpher.languagehandlers.hunmorph.IHunmorphLanguageHandler;
import com.github.szgabsz91.morpher.languagehandlers.hunmorph.converters.HunmorphLanguageHandlerConverter;
import com.github.szgabsz91.morpher.languagehandlers.hunmorph.impl.markov.FullMarkovModel;
import com.github.szgabsz91.morpher.languagehandlers.hunmorph.impl.markov.IMarkovModel;
import com.github.szgabsz91.morpher.languagehandlers.hunmorph.impl.model.HunmorphResult;
import com.github.szgabsz91.morpher.languagehandlers.hunmorph.protocolbuffers.HunmorphLanguageHandlerMessage;
import com.github.szgabsz91.morpher.core.io.Serializer;
import com.github.szgabsz91.morpher.core.model.AffixType;
import com.github.szgabsz91.morpher.core.model.FrequencyAwareWord;
import com.github.szgabsz91.morpher.core.model.FrequencyAwareWordPair;
import com.github.szgabsz91.morpher.core.model.Word;
import com.github.szgabsz91.morpher.core.services.Qualifier;
import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

/**
 * Hunmorph-Ocamorph based language handler implementation.
 *
 * @author szgabsz91
 */
@Qualifier(IHunmorphLanguageHandler.QUALIFIER)
public class HunmorphLanguageHandler implements ILanguageHandler<HunmorphLanguageHandlerMessage> {

    private static final AffixType UNKNOWN_POS = AffixType.of("/UNKNOWN");

    private final HunmorphWordProcessor hunmorphWordProcessor;
    private final HunmorphAnnotationTokenizer hunmorphAnnotationTokenizer;
    private Map<String, List<AnnotationTokenizerResult>> annotationTokenizerResultMap;
    private IMarkovModel markovModel;
    private IMarkovModel reversedMarkovModel;
    private Map<String, Set<AffixType>> lemmaMap;

    /**
     * Default constructor that initializes the language handler state.
     */
    public HunmorphLanguageHandler() {
        this(new HunmorphWordProcessor(), new FullMarkovModel(), new FullMarkovModel());
    }

    /**
     * Constructor that receives the {@link HunmorphWordProcessor} instance from outside.
     * @param hunmorphWordProcessor the {@link HunmorphWordProcessor} instance
     */
    HunmorphLanguageHandler(final HunmorphWordProcessor hunmorphWordProcessor) {
        this(hunmorphWordProcessor, new FullMarkovModel(), new FullMarkovModel());
    }

    /**
     * Constructor that receives the {@link HunmorphWordProcessor} and {@link IMarkovModel} instances from outside.
     * @param hunmorphWordProcessor the {@link HunmorphWordProcessor} instance
     * @param markovModel the {@link IMarkovModel} instance to use for inflection
     * @param reversedMarkovModel the {@link IMarkovModel} instance to use for morphological analysis
     */
    HunmorphLanguageHandler(
            final HunmorphWordProcessor hunmorphWordProcessor,
            final IMarkovModel markovModel,
            final IMarkovModel reversedMarkovModel) {
        this.hunmorphWordProcessor = hunmorphWordProcessor;
        this.hunmorphAnnotationTokenizer = new HunmorphAnnotationTokenizer();
        this.annotationTokenizerResultMap = new HashMap<>();
        this.markovModel = markovModel;
        this.reversedMarkovModel = reversedMarkovModel;
        this.lemmaMap = new HashMap<>();
    }

    /**
     * Returns the map of annotation tokenizer results.
     * @return the map of annotation tokenizer results
     */
    public Map<String, List<AnnotationTokenizerResult>> getAnnotationTokenizerResultMap() {
        return annotationTokenizerResultMap;
    }

    /**
     * Sets the map of annotation tokenizer results.
     * @param annotationTokenizerResultMap the map of annotation tokenizer results
     */
    public void setAnnotationTokenizerResultMap(
            final Map<String, List<AnnotationTokenizerResult>> annotationTokenizerResultMap) {
        this.annotationTokenizerResultMap = annotationTokenizerResultMap;
    }

    /**
     * Returns the underlying {@link IMarkovModel} instance to use for inflection.
     * @return the underlying {@link IMarkovModel} instance to use for inflection
     */
    public IMarkovModel getMarkovModel() {
        return markovModel;
    }

    /**
     * Sets the underlying {@link IMarkovModel} instance to use for inflection.
     * @param markovModel the underlying {@link IMarkovModel} instance to use for inflection
     */
    public void setMarkovModel(final IMarkovModel markovModel) {
        this.markovModel = markovModel;
    }

    /**
     * Returns the underlying {@link IMarkovModel} instance to use for morphological analysis.
     * @return the underlying {@link IMarkovModel} instance to use for morphological analysis
     */
    public IMarkovModel getReversedMarkovModel() {
        return reversedMarkovModel;
    }

    /**
     * Sets the underlying {@link IMarkovModel} instance to use for morphological analysis.
     * @param reversedMarkovModel the underlying {@link IMarkovModel} instance to use for morphological analysis
     */
    public void setReversedMarkovModel(final IMarkovModel reversedMarkovModel) {
        this.reversedMarkovModel = reversedMarkovModel;
    }

    /**
     * Returns the map of lemmas.
     * @return the map of lemmas
     */
    public Map<String, Set<AffixType>> getLemmaMap() {
        return lemmaMap;
    }

    /**
     * Sets the map of lemmas.
     * @param lemmaMap the map of lemmas
     */
    public void setLemmaMap(final Map<String, Set<AffixType>> lemmaMap) {
        this.lemmaMap = lemmaMap;
    }

    /**
     * Closes the internal {@link HunmorphWordProcessor} instance.
     */
    @Override
    public void close() {
        this.hunmorphWordProcessor.close();
    }

    /**
     * Learns the given {@link AnnotationTokenizerResult}s.
     * @param annotationTokenizerResults the map of {@link AnnotationTokenizerResult}s for each affix type
     */
    @Override
    public void learnAnnotationTokenizerResults(
            final Map<String, List<AnnotationTokenizerResult>> annotationTokenizerResults) {
        annotationTokenizerResults.forEach((lemma, results) -> {
            final List<AnnotationTokenizerResult> existingResults = this.annotationTokenizerResultMap.get(lemma);
            if (existingResults == null) {
                this.annotationTokenizerResultMap.put(lemma, results);
            }
            else {
                existingResults.addAll(results);
            }
        });
    }

    /**
     * Learns the given {@link AffixType} chains.
     *
     * {@inheritDoc}
     */
    @Override
    public void learnAffixTypeChains(final Set<List<AffixType>> affixTypeChains) {
        affixTypeChains.forEach(affixTypeChain -> {
            this.markovModel.add(affixTypeChain);

            final List<AffixType> reversedAffixTypeChain = IntStream.range(0, affixTypeChain.size())
                    .map(index -> affixTypeChain.size() - index - 1)
                    .mapToObj(affixTypeChain::get)
                    .collect(toList());
            this.reversedMarkovModel.add(reversedAffixTypeChain);
        });
    }

    /**
     * Adds the words in the given map to the set of known lemmas.
     * @param lemmaMap map containing the lemmas and their part of speech tags
     */
    @Override
    public void learnLemmas(final LemmaMap lemmaMap) {
        lemmaMap.entrySet()
                .stream()
                .map(entry -> Map.entry(entry.getKey().toString(), entry.getValue()))
                .forEach(entry -> {
                    final String lemma = entry.getKey();
                    final Set<AffixType> newAffixTypes = entry.getValue();
                    final Set<AffixType> existingAffixTypes =
                            this.lemmaMap.computeIfAbsent(lemma, k -> new HashSet<>());
                    existingAffixTypes.addAll(newAffixTypes);
                });
    }

    /**
     * Returns a list of {@link AnnotationTokenizerResult} for the given {@link Word}.
     * @param word the {@link Word} to analyze
     * @return a list of {@link AnnotationTokenizerResult}
     */
    public List<AnnotationTokenizerResult> analyzeInternally(final FrequencyAwareWord word) {
        return this.analyzeInternally(word, false);
    }

    /**
     * Returns a list of {@link AnnotationTokenizerResult} for the given {@link Word}.
     * @param word the {@link Word} to analyze
     * @param guess flag that indicates if unknown words should be analyzed or not
     * @return a list of {@link AnnotationTokenizerResult}
     */
    public List<AnnotationTokenizerResult> analyzeInternally(final FrequencyAwareWord word, final boolean guess) {
        final String wordString = word.getWord().toString();
        final Optional<HunmorphResult> optionalHunmorphResult = this.hunmorphWordProcessor.process(wordString, guess);
        return optionalHunmorphResult
                .map(hunmorphResult -> {
                    return hunmorphResult.getOutputLines()
                            .stream()
                            .map(line -> {
                                final AnnotationTokenizerResult result = this.hunmorphAnnotationTokenizer.tokenize(
                                        line,
                                        wordString,
                                        word.getFrequency()
                                );
                                if (result == null) {
                                    return null;
                                }
                                return this.hunmorphAnnotationTokenizer.preprocess(result);
                            })
                            .filter(Objects::nonNull)
                            .collect(Collectors.toList());
                })
                .orElse(Collections.emptyList());
    }

    /**
     * Analyzes the given word, updates the internal state and returns new word pairs.
     * @param word the word to analyze
     * @return the response that contains a map, where the keys are affix types and values are word pair sets
     */
    @Override
    public LanguageHandlerResponse analyze(final FrequencyAwareWord word) {
        final List<AnnotationTokenizerResult> annotationTokenizerResults = this.analyzeInternally(word);
        if (annotationTokenizerResults.isEmpty()) {
            return LanguageHandlerResponse.of(new HashMap<>());
        }
        final Map<AffixType, Set<FrequencyAwareWordPair>> affixTypeMap = process(annotationTokenizerResults);
        return LanguageHandlerResponse.of(affixTypeMap);
    }

    /**
     * Analyzes the given set of words, updates the internal state and returns new word pairs.
     * @param words the words to analyze
     * @return the response that contains a map, where the keys are affix types and values are word pair sets
     */
    @Override
    public LanguageHandlerResponse analyze(final Set<FrequencyAwareWord> words) {
        final List<AnnotationTokenizerResult> annotationTokenizerResults = words
                .stream()
                .flatMap(frequencyAwareWord -> {
                    return this.hunmorphWordProcessor.process(frequencyAwareWord.getWord().toString())
                            .map(hunmorphResult -> {
                                final String wordString = hunmorphResult.getGrammaticalForm();
                                return hunmorphResult.getOutputLines()
                                        .stream()
                                        .map(line -> {
                                            @SuppressWarnings("checkstyle:LineLength")
                                            final AnnotationTokenizerResult result =
                                                    this.hunmorphAnnotationTokenizer.tokenize(line, wordString, frequencyAwareWord.getFrequency());
                                            if (result == null) {
                                                return null;
                                            }
                                            return this.hunmorphAnnotationTokenizer.preprocess(result);
                                        })
                                        .filter(Objects::nonNull);
                            })
                            .orElse(Stream.empty());
                })
                .collect(toList());
        final Map<AffixType, Set<FrequencyAwareWordPair>> affixTypeMap = process(annotationTokenizerResults);
        return LanguageHandlerResponse.of(affixTypeMap);
    }

    /**
     * Returns the list of affix type candidates for the given affix types, sorted by relative frequency, in descending
     * order.
     *
     * {@inheritDoc}
     */
    @Override
    public List<ProbabilisticAffixType> getAnalysisCandidates(final List<AffixType> affixTypes) {
        return this.reversedMarkovModel.getCandidates(affixTypes);
    }

    /**
     * Returns a {@link ProbabilisticAffixType} that has an ending affix type and the conditional probability of ending
     * the given chain.
     *
     * {@inheritDoc}
     */
    @Override
    public ProbabilisticAffixType getEndingAnalysisCandidate(final List<AffixType> affixTypes) {
        return this.getAnalysisCandidates(affixTypes)
                .stream()
                .filter(probabilisticAffixType -> probabilisticAffixType.getAffixType().equals(IMarkovModel.END))
                .findFirst()
                .orElse(ProbabilisticAffixType.of(IMarkovModel.END, 0.0));
    }

    /**
     * Returns the part of speech tag of the given lemma, or an empty optional if the lemma is unknown.
     *
     * {@inheritDoc}
     */
    @Override
    public List<ProbabilisticAffixType> getPOSCandidates(final Word lemma) {
        final Set<AffixType> affixTypes = this.lemmaMap.get(lemma.toString());

        if (affixTypes == null) {
            return Collections.emptyList();
        }

        return affixTypes
                .stream()
                .map(this.markovModel::getProbabilityOfPOS)
                .collect(toList());
    }

    /**
     * Returns if the given affix type is a part of speech tag or not.
     *
     * {@inheritDoc}
     */
    @Override
    public boolean isPOS(final AffixType affixType) {
        return affixType.toString().startsWith("/");
    }

    /**
     * Returns if the given affix type chain is valid or not.
     *
     * {@inheritDoc}
     */
    @Override
    public boolean isAffixTypeChainValid(final List<AffixType> affixTypeChain) {
        return this.markovModel.isAffixTypeChainValid(affixTypeChain);
    }

    /**
     * Sorts the given set of affix types based on the previously learnt probabilities.
     *
     * {@inheritDoc}
     */
    @Override
    public List<AffixTypeChain> sortAffixTypes(final Word lemma, final Set<AffixType> affixTypes) {
        return this.markovModel.sortAffixTypes(affixTypes, this::isPOS);
    }

    /**
     * Creates a new {@link AffixTypeChain} containing the given affix types.
     *
     * {@inheritDoc}
     */
    @Override
    public AffixTypeChain calculateProbabilities(final List<AffixType> affixTypes) {
        final AffixTypeChain affixTypeChain = this.markovModel.calculateProbabilities(affixTypes);
        final List<ProbabilisticAffixType> probabilisticAffixTypes = affixTypeChain.getAffixTypes();
        final AffixType firstAffixType = probabilisticAffixTypes.get(0).getAffixType();

        if (!this.isPOS(firstAffixType)) {
            final ProbabilisticAffixType unknownPos = ProbabilisticAffixType.of(UNKNOWN_POS, 0.0);
            probabilisticAffixTypes.add(0, unknownPos);
        }

        return affixTypeChain;
    }

    /**
     * Returns the set of supported affix types.
     *
     * {@inheritDoc}
     */
    @Override
    public Set<AffixType> getSupportedAffixTypes() {
        return this.hunmorphAnnotationTokenizer.getSupportedAffixTypes()
                .stream()
                .map(AffixType::of)
                .collect(toSet());
    }

    /**
     * Converts the state to a message.
     * @return the message
     */
    @Override
    public HunmorphLanguageHandlerMessage toMessage() {
        final HunmorphLanguageHandlerConverter converter = new HunmorphLanguageHandlerConverter();
        return converter.convert(this);
    }

    /**
     * Loads the state from the given message.
     * @param message the message
     */
    @Override
    public void fromMessage(final HunmorphLanguageHandlerMessage message) {
        final HunmorphLanguageHandlerConverter converter = new HunmorphLanguageHandlerConverter();

        try (final HunmorphLanguageHandler hunmorphLanguageHandler = converter.convertBack(message)) {
            this.annotationTokenizerResultMap = hunmorphLanguageHandler.annotationTokenizerResultMap;
            this.markovModel = hunmorphLanguageHandler.markovModel;
            this.reversedMarkovModel = hunmorphLanguageHandler.reversedMarkovModel;
            this.lemmaMap = hunmorphLanguageHandler.lemmaMap;
        }
    }

    /**
     * Loads the state from the given message.
     * @param message the message
     * @throws InvalidProtocolBufferException if the provided message is not a {@link HunmorphLanguageHandlerMessage}
     */
    @Override
    public void fromMessage(final Any message) throws InvalidProtocolBufferException {
        if (!message.is(HunmorphLanguageHandlerMessage.class)) {
            throw new InvalidProtocolBufferException(
                    "The provided message is not a HunmorphLanguageHandlerMessage: " + message
            );
        }

        final HunmorphLanguageHandlerMessage hunmorphLanguageHandlerMessage = message.unpack(
                HunmorphLanguageHandlerMessage.class
        );
        this.fromMessage(hunmorphLanguageHandlerMessage);
    }

    /**
     * Saves the language handler state to the given file.
     * @param file the file to save the state to
     * @throws IOException if the file cannot be written
     */
    @Override
    public void saveTo(final Path file) throws IOException {
        final HunmorphLanguageHandlerConverter converter = new HunmorphLanguageHandlerConverter();
        final Serializer<HunmorphLanguageHandler, HunmorphLanguageHandlerMessage> serializer =
                new Serializer<>(converter, this);
        serializer.serialize(this, file);
    }

    /**
     * Loads the language handler from the given file.
     * @param file the file with the state of the language handler
     * @throws IOException if the file cannot be read
     */
    @Override
    public void loadFrom(final Path file) throws IOException {
        final HunmorphLanguageHandlerConverter converter = new HunmorphLanguageHandlerConverter();
        final Serializer<HunmorphLanguageHandler, HunmorphLanguageHandlerMessage> serializer =
                new Serializer<>(converter, this);

        try (final HunmorphLanguageHandler hunmorphLanguageHandler = serializer.deserialize(file)) {
            this.annotationTokenizerResultMap = hunmorphLanguageHandler.annotationTokenizerResultMap;
            this.markovModel = hunmorphLanguageHandler.markovModel;
            this.reversedMarkovModel = hunmorphLanguageHandler.reversedMarkovModel;
            this.lemmaMap = hunmorphLanguageHandler.lemmaMap;
        }
    }

    private Map<AffixType, Set<FrequencyAwareWordPair>> process(
            final List<AnnotationTokenizerResult> annotationTokenizerResults) {
        final Map<AffixType, Set<FrequencyAwareWordPair>> affixTypeMap = new HashMap<>();
        final Map<String, List<AnnotationTokenizerResult>> processedRecordMap =
                getProcessedRecordMap(annotationTokenizerResults);

        for (final Map.Entry<String, List<AnnotationTokenizerResult>> entry : processedRecordMap.entrySet()) {
            final String lemma = entry.getKey();
            final List<AnnotationTokenizerResult> annotationTokenizerResultList = entry.getValue();
            final List<AnnotationTokenizerResult> existingAnnotationTokenizerResultList =
                    this.annotationTokenizerResultMap.computeIfAbsent(lemma, k -> new ArrayList<>());
            final Set<Integer> newIndices =
                    addAll(existingAnnotationTokenizerResultList, annotationTokenizerResultList);

            if (newIndices.isEmpty()) {
                continue;
            }

            for (final int newIndex : newIndices) {
                final AnnotationTokenizerResult annotationTokenizerResult =
                        existingAnnotationTokenizerResultList.get(newIndex);
                final String word = annotationTokenizerResult.getGrammaticalForm();
                final int frequency = annotationTokenizerResult.getFrequency();

                if (word.equals(lemma)) {
                    continue;
                }

                final List<AffixType> affixTypes = annotationTokenizerResult.getAffixTypes();
                this.markovModel.add(affixTypes);
                final List<AffixType> reversedAffixTypes = IntStream.range(0, affixTypes.size())
                        .map(index -> affixTypes.size() - index - 1)
                        .mapToObj(affixTypes::get)
                        .collect(toList());
                this.reversedMarkovModel.add(reversedAffixTypes);

                final Set<AffixType> posSet = this.lemmaMap.computeIfAbsent(lemma, k -> new HashSet<>());
                posSet.add(affixTypes.get(0));

                if (affixTypes.size() == 2) {
                    final FrequencyAwareWordPair wordPair = FrequencyAwareWordPair.of(lemma, word, frequency);
                    put(affixTypeMap, affixTypes.get(affixTypes.size() - 1), wordPair);
                }

                for (int index = 0; index < existingAnnotationTokenizerResultList.size(); index++) {
                    if (index == newIndex) {
                        continue;
                    }

                    final AnnotationTokenizerResult existingAnnotationTokenizerResult =
                            existingAnnotationTokenizerResultList.get(index);
                    final List<AffixType> existingAffixTypes = existingAnnotationTokenizerResult.getAffixTypes();

                    if (Math.abs(existingAffixTypes.size() - affixTypes.size()) != 1) {
                        continue;
                    }

                    final AnnotationTokenizerResult shorterAnnotationTokenizerResult =
                            existingAffixTypes.size() < affixTypes.size() ?
                            existingAnnotationTokenizerResult :
                            annotationTokenizerResult;
                    final AnnotationTokenizerResult longerAnnotationTokenizerResult =
                            existingAffixTypes.size() < affixTypes.size() ?
                            annotationTokenizerResult :
                            existingAnnotationTokenizerResult;
                    final List<AffixType> shorterAffixTypes = shorterAnnotationTokenizerResult.getAffixTypes();
                    final List<AffixType> longerAffixTypes = longerAnnotationTokenizerResult.getAffixTypes();

                    if (longerAffixTypes.subList(0, longerAffixTypes.size() - 1).equals(shorterAffixTypes)) {
                        final FrequencyAwareWordPair wordPair = FrequencyAwareWordPair.of(
                                shorterAnnotationTokenizerResult.getGrammaticalForm(),
                                longerAnnotationTokenizerResult.getGrammaticalForm(),
                                frequency
                        );
                        put(affixTypeMap, longerAffixTypes.get(longerAffixTypes.size() - 1), wordPair);
                    }
                }
            }
        }

        return affixTypeMap;
    }

    private Map<String, List<AnnotationTokenizerResult>> getProcessedRecordMap(
            final List<AnnotationTokenizerResult> annotationTokenizerResults) {
        return annotationTokenizerResults
                .stream()
                .map(annotationTokenizerResult -> {
                    final String word = annotationTokenizerResult.getGrammaticalForm();
                    final String lemma = annotationTokenizerResult.getLemma();
                    final List<AffixType> affixTypes = annotationTokenizerResult.getAffixTypes();
                    if (affixTypes.size() == 1 || word.equals(lemma)) {
                        return null;
                    }
                    return annotationTokenizerResult;
                })
                .filter(Objects::nonNull)
                .collect(groupingBy(AnnotationTokenizerResult::getLemma));
    }

    private static <K, V> void put(final Map<K, Set<V>> map, final K affixType, final V wordPair) {
        final Set<V> wordPairSet = map.computeIfAbsent(affixType, k -> new HashSet<>());
        wordPairSet.add(wordPair);
    }

    private static <T extends Comparable<T>> Set<Integer> addAll(final List<T> target, final List<T> source) {
        if (target.isEmpty()) {
            target.addAll(source);
            Collections.sort(target);
            return IntStream.range(0, target.size())
                    .boxed()
                    .collect(toSet());
        }

        source.removeAll(target);
        if (source.isEmpty()) {
            return Collections.emptySet();
        }

        target.addAll(source);
        Collections.sort(target);

        final Set<Integer> indices = new HashSet<>();
        for (int i = 0; i < target.size(); i++) {
            final T item = target.get(i);

            if (source.contains(item)) {
                indices.add(i);
            }
        }
        return indices;
    }

}
