/*
 *                    GNU LESSER GENERAL PUBLIC LICENSE
 *                        Version 3, 29 June 2007
 *
 *  Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *
 *   This version of the GNU Lesser General Public License incorporates
 * the terms and conditions of version 3 of the GNU General Public
 * License, supplemented by the additional permissions listed below.
 *
 *   0. Additional Definitions.
 *
 *   As used herein, "this License" refers to version 3 of the GNU Lesser
 * General Public License, and the "GNU GPL" refers to version 3 of the GNU
 * General Public License.
 *
 *   "The Library" refers to a covered work governed by this License,
 * other than an Application or a Combined Work as defined below.
 *
 *   An "Application" is any work that makes use of an interface provided
 * by the Library, but which is not otherwise based on the Library.
 * Defining a subclass of a class defined by the Library is deemed a mode
 * of using an interface provided by the Library.
 *
 *   A "Combined Work" is a work produced by combining or linking an
 * Application with the Library.  The particular version of the Library
 * with which the Combined Work was made is also called the "Linked
 * Version".
 *
 *   The "Minimal Corresponding Source" for a Combined Work means the
 * Corresponding Source for the Combined Work, excluding any source code
 * for portions of the Combined Work that, considered in isolation, are
 * based on the Application, and not on the Linked Version.
 *
 *   The "Corresponding Application Code" for a Combined Work means the
 * object code and/or source code for the Application, including any data
 * and utility programs needed for reproducing the Combined Work from the
 * Application, but excluding the System Libraries of the Combined Work.
 *
 *   1. Exception to Section 3 of the GNU GPL.
 *
 *   You may convey a covered work under sections 3 and 4 of this License
 * without being bound by section 3 of the GNU GPL.
 *
 *   2. Conveying Modified Versions.
 *
 *   If you modify a copy of the Library, and, in your modifications, a
 * facility refers to a function or data to be supplied by an Application
 * that uses the facility (other than as an argument passed when the
 * facility is invoked), then you may convey a copy of the modified
 * version:
 *
 *    a) under this License, provided that you make a good faith effort to
 *    ensure that, in the event an Application does not supply the
 *    function or data, the facility still operates, and performs
 *    whatever part of its purpose remains meaningful, or
 *
 *    b) under the GNU GPL, with none of the additional permissions of
 *    this License applicable to that copy.
 *
 *   3. Object Code Incorporating Material from Library Header Files.
 *
 *   The object code form of an Application may incorporate material from
 * a header file that is part of the Library.  You may convey such object
 * code under terms of your choice, provided that, if the incorporated
 * material is not limited to numerical parameters, data structure
 * layouts and accessors, or small macros, inline functions and templates
 * (ten or fewer lines in length), you do both of the following:
 *
 *    a) Give prominent notice with each copy of the object code that the
 *    Library is used in it and that the Library and its use are
 *    covered by this License.
 *
 *    b) Accompany the object code with a copy of the GNU GPL and this license
 *    document.
 *
 *   4. Combined Works.
 *
 *   You may convey a Combined Work under terms of your choice that,
 * taken together, effectively do not restrict modification of the
 * portions of the Library contained in the Combined Work and reverse
 * engineering for debugging such modifications, if you also do each of
 * the following:
 *
 *    a) Give prominent notice with each copy of the Combined Work that
 *    the Library is used in it and that the Library and its use are
 *    covered by this License.
 *
 *    b) Accompany the Combined Work with a copy of the GNU GPL and this license
 *    document.
 *
 *    c) For a Combined Work that displays copyright notices during
 *    execution, include the copyright notice for the Library among
 *    these notices, as well as a reference directing the user to the
 *    copies of the GNU GPL and this license document.
 *
 *    d) Do one of the following:
 *
 *        0) Convey the Minimal Corresponding Source under the terms of this
 *        License, and the Corresponding Application Code in a form
 *        suitable for, and under terms that permit, the user to
 *        recombine or relink the Application with a modified version of
 *        the Linked Version to produce a modified Combined Work, in the
 *        manner specified by section 6 of the GNU GPL for conveying
 *        Corresponding Source.
 *
 *        1) Use a suitable shared library mechanism for linking with the
 *        Library.  A suitable mechanism is one that (a) uses at run time
 *        a copy of the Library already present on the user's computer
 *        system, and (b) will operate properly with a modified version
 *        of the Library that is interface-compatible with the Linked
 *        Version.
 *
 *    e) Provide Installation Information, but only if you would otherwise
 *    be required to provide such information under section 6 of the
 *    GNU GPL, and only to the extent that such information is
 *    necessary to install and execute a modified version of the
 *    Combined Work produced by recombining or relinking the
 *    Application with a modified version of the Linked Version. (If
 *    you use option 4d0, the Installation Information must accompany
 *    the Minimal Corresponding Source and Corresponding Application
 *    Code. If you use option 4d1, you must provide the Installation
 *    Information in the manner specified by section 6 of the GNU GPL
 *    for conveying Corresponding Source.)
 *
 *   5. Combined Libraries.
 *
 *   You may place library facilities that are a work based on the
 * Library side by side in a single library together with other library
 * facilities that are not Applications and are not covered by this
 * License, and convey such a combined library under terms of your
 * choice, if you do both of the following:
 *
 *    a) Accompany the combined library with a copy of the same work based
 *    on the Library, uncombined with any other library facilities,
 *    conveyed under the terms of this License.
 *
 *    b) Give prominent notice with the combined library that part of it
 *    is a work based on the Library, and explaining where to find the
 *    accompanying uncombined form of the same work.
 *
 *   6. Revised Versions of the GNU Lesser General Public License.
 *
 *   The Free Software Foundation may publish revised and/or new versions
 * of the GNU Lesser General Public License from time to time. Such new
 * versions will be similar in spirit to the present version, but may
 * differ in detail to address new problems or concerns.
 *
 *   Each version is given a distinguishing version number. If the
 * Library as you received it specifies that a certain numbered version
 * of the GNU Lesser General Public License "or any later version"
 * applies to it, you have the option of following the terms and
 * conditions either of that published version or of any later version
 * published by the Free Software Foundation. If the Library as you
 * received it does not specify a version number of the GNU Lesser
 * General Public License, you may choose any version of the GNU Lesser
 * General Public License ever published by the Free Software Foundation.
 *
 *   If the Library as you received it specifies that a proxy can decide
 * whether future versions of the GNU Lesser General Public License shall
 * apply, that proxy's public statement of acceptance of any version is
 * permanent authorization for you to choose that version for the
 * Library.
 */
package com.github.szgabsz91.morpher.languagehandlers.hunmorph.impl;

import com.github.szgabsz91.morpher.core.model.AffixType;
import com.github.szgabsz91.morpher.languagehandlers.api.model.AnnotationTokenizerResult;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static java.util.Comparator.comparingInt;
import static java.util.stream.Collectors.toList;

/**
 * Tokenizes Hunmorph-Ocamorph token strings.
 *
 * @author szgabsz91
 */
public class HunmorphAnnotationTokenizer {

    /**
     * The list of known tokens.
     */
    public static final List<String> KNOWN_TOKENS;

    static {
        final List<String> knownTokens = new ArrayList<>(List.of(
                "/ADJ", "/ADV", "/ART", "/CONJ", "/DET", "/NOUN", "/NUM", "/ONO", "/POSTP", "/PREP", "/PUNCT",
                "/UTT-INT", "/VERB", "/PREV", "<PLUR>", "<PLUR<FAM>>", "<POSS<1>>", "<POSS<2>>", "<POSS>",
                "<POSS<1><PLUR>>", "<POSS<2><PLUR>>", "<POSS<PLUR>>", "<PLUR><POSS<1>>", "<PLUR><POSS<2>>",
                "<PLUR><POSS>", "<PLUR><POSS<1><PLUR>>", "<PLUR><POSS<2><PLUR>>", "<PLUR><POSS<PLUR>>", "<ANP>",
                "<ANP<PLUR>>", "<PLUR<ANP>>", "<CAS<ACC>>", "<CAS<DAT>>", "<CAS<INS>>", "<CAS<CAU>>", "<CAS<TRA>>",
                "<CAS<SUE>>", "<CAS<SBL>>", "<CAS<DEL>>", "<CAS<INE>>", "<CAS<ELA>>", "<CAS<ILL>>", "<CAS<ADE>>",
                "<CAS<ALL>>", "<CAS<ABL>>", "<CAS<TEM>>", "<CAS<TER>>", "<CAS<FOR>>", "<CAS<ESS>>", "<MODAL>",
                "<SUBJUNC-IMP><PERS<1>>", "<SUBJUNC-IMP><PERS<2>>", "<SUBJUNC-IMP>", "<SUBJUNC-IMP><PERS<1>><PLUR>",
                "<SUBJUNC-IMP><PERS<2>><PLUR>", "<SUBJUNC-IMP><PLUR>", "<SUBJUNC-IMP><PERS<1>><DEF>",
                "<SUBJUNC-IMP><PERS<1<OBJ<2>>>>", "<SUBJUNC-IMP><PERS<2>><DEF>", "<SUBJUNC-IMP><DEF>",
                "<SUBJUNC-IMP><PERS<1>><PLUR><DEF>", "<SUBJUNC-IMP><PERS<2>><PLUR><DEF>", "<SUBJUNC-IMP><PLUR><DEF>",
                "<COND>", "<COND><PERS<1>>", "<COND><PERS<2>>", "<COND>", "<COND><PERS<1>><PLUR>",
                "<COND><PERS<2>><PLUR>", "<COND><PLUR>", "<COND><PERS<1>><DEF>", "<COND><PERS<1<OBJ<2>>>>",
                "<COND><PERS<2>><DEF>", "<COND><DEF>", "<COND><PERS<1>><PLUR><DEF>", "<COND><PERS<2>><PLUR><DEF>",
                "<COND><PLUR><DEF>", "<COND<PAST>>", "<PAST><PERS<1>>", "<PAST><PERS<2>>", "<PAST>",
                "<PAST><PERS<1>><PLUR>", "<PAST><PERS<2>><PLUR>", "<PAST><PLUR>", "<PAST><PERS<1>><DEF>",
                "<PAST><PERS<2>><DEF>", "<PAST><PERS<1<OBJ<2>>>>", "<PAST><DEF>", "<PAST><PERS<1>><PLUR><DEF>",
                "<PAST><PERS<2>><PLUR><DEF>", "<PAST><PLUR><DEF>", "<FUT>", "<PERS<1>>", "<PERS<1<OBJ<2>>>>",
                "<PERS<2>>", "<PERS>", "<PERS<1>><PLUR>", "<PERS<2>><PLUR>", "<VPLUR>", "<PERS<1>><DEF>",
                "<PERS<2>><DEF>", "<DEF>", "<PERS<1>><PLUR><DEF>", "<PERS<2>><PLUR><DEF>", "<PLUR><DEF>", "<INF>",
                "<INF><PERS<1>>", "<INF><PERS<2>>", "<INF><PERS>", "<INF><PERS<1>><PLUR>", "<INF><PERS<2>><PLUR>",
                "<INF><PERS><PLUR>", "[FREQ]", "[MEDIAL]", "[CAUS]", "[DESID]", "[PART]", "[PERF_PART]",
                "[IMPERF_PART]", "[FUT_PART]", "[NEG_PERF_PART]", "[GERUND]", "[NEG_MODAL_PART]", "[MODAL_PART]",
                "[REG_ACT]", "[ABSTRACT]", "[MRS]", "[DIMIN]", "[ATTRIB]", "[ATTRIB]", "[MET_ATTRIB]",
                "[MET_ATTRIB][MANNER]", "[INAL_ATTRIB]", "[NEG_ATTRIB]", "[TYPE1]", "[TYPE2]", "[TYPE3]", "[TYPE4]",
                "[TYPE_RANK]", "[NEG_ATTRIB2]", "[LOC_INE]", "[QUANTITY]", "[ESS_FOR]", "[COM]", "[PERIOD1]",
                "[PERIOD2]", "[ACT]", "[ACT2]", "[COMPAR]", "[SUPERLAT]", "[SUPERSUPERLAT]", "[COMPAR_DESIGN]",
                "[SUPERLAT_DESIGN]", "[SUPERSUPERLAT_DESIGN]", "[MANNER]", "[INTRANS_RESULT]", "[TRANS_RESULT]",
                "[MULTIPL-ITER]", "[ITER_ATTRIB]", "[MULTIPL_ATTRIB]", "[MULTIPL]", "[AGGREG]", "[FRACT]", "[ORD]",
                "[ORD-ITER]", "[ORD-ITER-ACCOMPL]", "[DATE]", "<POSTP<ALÁ>>", "<POSTP<ALATT>>", "<POSTP<ALÓL>>",
                "<POSTP<ÁLTAL>>", "<POSTP<ELÉ>>", "<POSTP<ELÉBE>>", "<POSTP<ELLEN>>", "<POSTP<ELLENÉRE>>",
                "<POSTP<ELŐL>>", "<POSTP<ELŐTT>>", "<POSTP<FELETT>>", "<POSTP<FELÉ>>", "<POSTP<FELŐL>>",
                "<POSTP<FELÜL>>", "<POSTP<FÖLÉ>>", "<POSTP<FÖLIBE>>", "<POSTP<FÖLÜL>>", "<POSTP<HELYETT>>",
                "<POSTP<IRÁNT>>", "<POSTP<KÖRÉ>>", "<POSTP<KÖRÖTT>>", "<POSTP<KÖRÜL>>", "<POSTP<KÖZBEN>>",
                "<POSTP<KÖZÉ>>", "<POSTP<KÖZIBE>>", "<POSTP<KÖZÖTT>>", "<POSTP<KÖZÜL>>", "<POSTP<LÉTÉRE>>",
                "<POSTP<MELLETT>>", "<POSTP<MELLÉ>>", "<POSTP<MELLŐL>>", "<POSTP<MIATT>>", "<POSTP<MÖGÉ>>",
                "<POSTP<MÖGÖTT>>", "<POSTP<MÖGÜL>>", "<POSTP<NÉLKÜL>>", "<POSTP<ÓTA>>", "<POSTP<RÉSZÉRE>>",
                "<POSTP<RÉSZÉRŐL>>", "<POSTP<SZÁMÁRA>>", "<POSTP<SZERINT>>", "<POSTP<UTÁN>>", "<POSTP<VÉGBŐL>>",
                "<POSTP<VÉGETT>>", "<POSTP<VÉGRE>>", "<PREVERB<ABBA>>", "<PREVERB<AGYON>>", "<PREVERB<ALUL>>",
                "<PREVERB<ALÁ>>", "<PREVERB<ALÁBB>>", "<PREVERB<ARRÉBB>>", "<PREVERB<BE>>", "<PREVERB<BELE>>",
                "<PREVERB<BELJEBB>>", "<PREVERB<BENN>>", "<PREVERB<BENNE>>", "<PREVERB<BENT>>", "<PREVERB<BÉ>>",
                "<PREVERB<BÉKÉN>>", "<PREVERB<CSERBEN>>", "<PREVERB<EGYBE>>", "<PREVERB<EGYET>>", "<PREVERB<EGYÜTT>>",
                "<PREVERB<EL>>", "<PREVERB<ELLEN>>", "<PREVERB<ELLENT>>", "<PREVERB<ELÉ>>", "<PREVERB<ELŐ>>",
                "<PREVERB<ELŐRE>>", "<PREVERB<FEL>>", "<PREVERB<FELJEBB>>", "<PREVERB<FELÜL>>", "<PREVERB<FENN>>",
                "<PREVERB<FÉLBE>>", "<PREVERB<FÉLRE>>", "<PREVERB<FÖL>>", "<PREVERB<FÖLDET>>", "<PREVERB<FÖLJEBB>>",
                "<PREVERB<FÖLÉ>>", "<PREVERB<FÖLÜL>>", "<PREVERB<FÖNN>>", "<PREVERB<HAZA>>", "<PREVERB<HELYBEN>>",
                "<PREVERB<HELYRE>>", "<PREVERB<HELYT>>", "<PREVERB<HOPPON>>", "<PREVERB<HOZZÁ>>", "<PREVERB<HÁTRA>>",
                "<PREVERB<IDE>>", "<PREVERB<IDÉBB>>", "<PREVERB<ITT>>", "<PREVERB<JÓL>>", "<PREVERB<JÓT>>",
                "<PREVERB<JÓVÁ>>", "<PREVERB<KARBAN>>", "<PREVERB<KERESZTÜL>>", "<PREVERB<KETTÉ>>", "<PREVERB<KI>>",
                "<PREVERB<KIJJEBB>>", "<PREVERB<KINN>>", "<PREVERB<KINT>>", "<PREVERB<KÉTSÉGBE>>", "<PREVERB<KÖLCSÖN>>",
                "<PREVERB<KÖRBE>>", "<PREVERB<KÖRÜL>>", "<PREVERB<KÖZBE>>", "<PREVERB<KÖZBEN>>", "<PREVERB<KÖZRE>>",
                "<PREVERB<KÖZZÉ>>", "<PREVERB<KÖZÉ>>", "<PREVERB<KÜLÖN>>", "<PREVERB<LE>>", "<PREVERB<LEJJEBB>>",
                "<PREVERB<LÉTRE>>", "<PREVERB<MEG>>", "<PREVERB<MELLÉ>>", "<PREVERB<MÖGÉ>>", "<PREVERB<NAGYOT>>",
                "<PREVERB<NEKI>>", "<PREVERB<NYILVÁN>>", "<PREVERB<NYUGTON>>", "<PREVERB<ODA>>", "<PREVERB<ODÁBB>>",
                "<PREVERB<ODÉBB>>", "<PREVERB<OTT>>", "<PREVERB<POFON>>", "<PREVERB<PÓRUL>>", "<PREVERB<RAJT>>",
                "<PREVERB<RAJTA>>", "<PREVERB<RENDRE>>", "<PREVERB<RÁ>>", "<PREVERB<SZABADJÁRA>>",
                "<PREVERB<SZABADLÁBRA>>", "<PREVERB<SZEMBE>>", "<PREVERB<SZEMRE>>", "<PREVERB<SZEMÜGYRE>>",
                "<PREVERB<SZERTE>>", "<PREVERB<SZÁMBA>>", "<PREVERB<SZÉJJEL>>", "<PREVERB<SZÉT>>",
                "<PREVERB<SZÖRNYET>>", "<PREVERB<SÍKRA>>", "<PREVERB<TELE>>", "<PREVERB<TELI>>",
                "<PREVERB<TORKIG>>", "<PREVERB<TOVA>>", "<PREVERB<TOVÁBB>>", "<PREVERB<TUDOMÁSUL>>",
                "<PREVERB<TUDTUL>>", "<PREVERB<TÖNKRE>>", "<PREVERB<TÚL>>", "<PREVERB<UTOL>>", "<PREVERB<UTÁN>>",
                "<PREVERB<UTÁNA>>", "<PREVERB<UTÓL>>", "<PREVERB<VISSZA>>", "<PREVERB<VISZONT>>", "<PREVERB<VÉGBE>>",
                "<PREVERB<VÉGHEZ>>", "<PREVERB<VÉGIG>>", "<PREVERB<VÉGRE>>", "<PREVERB<ZOKON>>", "<PREVERB<ÁLTAL>>",
                "<PREVERB<ÁT>>", "<PREVERB<ÉSZHEZ>>", "<PREVERB<ÉSZRE>>", "<PREVERB<ÓVA>>", "<PREVERB<ÖSSZE>>",
                "<PREVERB<ÖSSZÉBB>>", "<PREVERB<ÚJJÁ>>", "<PREVERB<ÚJRA>>", "<PREVERB<ÚTBA>>", "<PREVERB<ŐRIZETLEN>>",
                "/ímél"
        ));
        knownTokens.sort(comparingInt(String::length).reversed());
        KNOWN_TOKENS = Collections.unmodifiableList(knownTokens);
    }

    /**
     * Tokenizes the given expression and grammatical form and returns the result.
     * @param expression the expression containing the tokens
     * @param grammaticalForm the grammatical form
     * @param frequency the frequency
     * @return the {@link AnnotationTokenizerResult} instance
     */
    public AnnotationTokenizerResult tokenize(
            final String expression,
            final String grammaticalForm,
            final int frequency) {
        String preprocessedExpression = preprocessExpression(expression);
        if (preprocessedExpression.contains("?")) {
            preprocessedExpression = preprocessedExpression.replace('?', '/');
        }
        final int firstSlashIndex = preprocessedExpression.indexOf('/');
        final String lemma = preprocessedExpression.substring(0, firstSlashIndex);
        final AnnotationTokenizerResult result =
                new AnnotationTokenizerResult(expression, grammaticalForm, lemma, frequency);
        preprocessedExpression = preprocessedExpression.substring(firstSlashIndex);

        if ("/PREV".equals(preprocessedExpression) || expression.startsWith("sapiens")) {
            return null;
        }

        final StringBuilder unknownTokenBuilder = new StringBuilder();

        for (int i = 0; i < preprocessedExpression.length(); i++) {
            String token = null;

            for (final String knownToken : KNOWN_TOKENS) {
                if (i + knownToken.length() > preprocessedExpression.length()) {
                    continue;
                }
                final String tokenCandidate = preprocessedExpression.substring(i, i + knownToken.length());
                if (tokenCandidate.equals(knownToken)) {
                    token = tokenCandidate;
                    break;
                }
            }

            if (preprocessedExpression.substring(i).startsWith("[SUPER-SUPERLAT]")) {
                preprocessedExpression =
                        preprocessedExpression.substring(0, preprocessedExpression.indexOf("[SUPER-SUPERLAT]")) +
                                "[SUPERSUPERLAT]" +
                                preprocessedExpression.substring(
                                        preprocessedExpression.indexOf("[SUPER-SUPERLAT]") + "[SUPER-SUPERLAT]".length()
                                );
                token = "[SUPERSUPERLAT]";
            }

            if (token != null) {
                if (unknownTokenBuilder.length() > 0) {
                    throw new IllegalStateException("Unknown token found: " + unknownTokenBuilder.toString());
                }

                result.addAffixType(AffixType.of(token));
                i += token.length() - 1;
            }
            else {
                unknownTokenBuilder.append(preprocessedExpression.charAt(i));
            }
        }

        if (unknownTokenBuilder.length() > 0) {
            throw new IllegalStateException("Unknown token found: " + unknownTokenBuilder.toString());
        }

        return result;
    }

    public Set<String> getSupportedAffixTypes() {
        return new HashSet<>(KNOWN_TOKENS);
    }

    private static String preprocessExpression(final String expression) {
        if (!expression.contains("+")) {
            return expression;
        }

        final String[] expressionParts = expression.split("\\+");
        final String preverb = expressionParts[0].split("/")[0];
        final String secondExpression = expressionParts[1];
        final String lemma = secondExpression.substring(0, secondExpression.indexOf('/'));
        final String preverbToken = String.format("<PREVERB<%s>>", preverb.toUpperCase());
        final String secondTokens = secondExpression.substring(secondExpression.indexOf('/'));
        final int separatorIndex1 = secondTokens.indexOf('<');
        final int separatorIndex2 = secondTokens.indexOf('[');
        final int separatorIndex = combineSeparatorIndices(separatorIndex1, separatorIndex2);

        if (separatorIndex == -1) {
            return lemma + secondTokens + preverbToken;
        }

        final String secondTokensStart = secondTokens.substring(0, separatorIndex);
        final String secondTokensEnd = secondTokens.substring(separatorIndex);
        return lemma + secondTokensStart + preverbToken + secondTokensEnd;
    }

    /**
     * Preprocesses the given result and returns the processed version.
     * @param originalResult the original result
     * @return the processed result
     */
    public AnnotationTokenizerResult preprocess(final AnnotationTokenizerResult originalResult) {
        final AnnotationTokenizerResult result = new AnnotationTokenizerResult(
                originalResult.getExpression(),
                originalResult.getGrammaticalForm().toLowerCase().trim(),
                originalResult.getLemma().toLowerCase().trim(),
                originalResult.getFrequency()
        );
        final List<AffixType> affixTypes = originalResult.getAffixTypes()
                .stream()
                .filter(affixType -> !"/ímél".equals(affixType.toString()))
                .map(affixType -> {
                    if ("<PLUR<ANP>>".equals(affixType.toString())) {
                        return AffixType.of("<ANP<PLUR>>");
                    }

                    return affixType;
                })
                .collect(toList());
        AffixType currentPOS = null;
        for (final AffixType affixType : affixTypes) {
            final AffixType pos = getPOS(affixType);
            if (pos != null) {
                currentPOS = pos;
                if (result.getAffixTypes().isEmpty()) {
                    result.addAffixType(affixType);
                }
            }
            else {
                if ("<PLUR>".equals(affixType.toString()) && "VERB".equals(currentPOS.toString())) {
                    result.addAffixType(AffixType.of("<VPLUR>"));
                }
                else {
                    result.addAffixType(affixType);
                }
            }
        }
        return result;
    }

    private static AffixType getPOS(final AffixType affixType) {
        if (affixType.toString().startsWith("/")) {
            return AffixType.of(affixType.toString().substring(1));
        }
        return null;
    }

    private static int combineSeparatorIndices(final int index1, final int index2) {
        if (index1 == -1) {
            return index2;
        }

        if (index2 == -1) {
            return index1;
        }

        return Math.min(index1, index2);
    }

}
