/*
 * Copyright 2016 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.mariadb.embedded;

import com.mariadb.embedded.exception.MariadbServerException;

import ch.vorburger.exec.ManagedProcessException;
import ch.vorburger.mariadb4j.DB;
import ch.vorburger.mariadb4j.DBConfiguration;
import ch.vorburger.mariadb4j.DBConfigurationBuilder;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import lombok.NonNull;
import lombok.Synchronized;

/**
 * Embedded MariaDB server
 */
public class MariadbServer {
	
	/**
	 * Lock object
	 */
	private final Object lock = new Object();
	
	/**
	 * {@link DBConfiguration}
	 */
	private DBConfiguration config;
	
	/**
	 * {@link DB}
	 */
	private DB delegate;
	
	/**
	 * Get running port
	 * 
	 * @return running port
	 */
	public int getPort() {
		
		return this.config.getPort();
	}
	
	/**
	 * Create JDBC URL
	 * 
	 * @param schema schema
	 * @return JDBC URL
	 */
	public String createJdbcUrl(String schema) {
		
		return DBConfigurationBuilder.newBuilder().setPort(this.getPort()).getURL(schema);
	}
	
	/**
	 * Constructor
	 * 
	 * @param config {@link DBConfiguration}
	 */
	protected MariadbServer(@NonNull DBConfiguration config) {
		
		this.config = config;
	}
	
	/**
	 * Get {@link Builder}
	 * 
	 * @return {@link Builder}
	 */
	public static Builder builder() {
		
		return new Builder();
	}
	
	/**
	 * Start server
	 * 
	 * @return {@link MariadbServer}
	 * @throws MariadbServerException if failed to start
	 */
	@Synchronized("lock")
	public MariadbServer start() throws MariadbServerException {
		
		if (this.isRunning()) {
			
			throw new MariadbServerException("Server is already running");
		}
		
		try {
			
			DB delegate = DB.newEmbeddedDB(this.config);
			delegate.start();
			
			this.delegate = delegate;
			
			return this;
		}
		catch (ManagedProcessException e) {
			
			throw new MariadbServerException("Failed to start server", e);
		}
	}
	
	/**
	 * Stop server
	 * 
	 * @return {@link MariadbServer}
	 * @throws MariadbServerException if failed to stop
	 */
	@Synchronized("lock")
	public MariadbServer stop() throws MariadbServerException {
		
		if (!this.isRunning()) {
			
			throw new MariadbServerException("Server is not running");
		}
		
		try {
			
			this.delegate.stop();
			this.delegate = null;
			
			return this;
		}
		catch (ManagedProcessException e) {
			
			throw new MariadbServerException("Failed to stop server", e);
		}
	}
	
	/**
	 * Check server is running
	 * 
	 * @return {@code true} if server is running
	 */
	@Synchronized("lock")
	public boolean isRunning() {
		
		return this.delegate != null;
	}
	
	/**
	 * Builder
	 */
	@NoArgsConstructor(access = AccessLevel.PROTECTED)
	public static class Builder {
		
		/**
		 * {@link DBConfigurationBuilder}
		 */
		private DBConfigurationBuilder delegate = DBConfigurationBuilder.newBuilder();
		
		/**
		 * Set port number
		 * 
		 * @param port port number
		 * @return {@link Builder}
		 */
		public Builder setPort(int port) {
			
			this.delegate.setPort(port);
			
			return this;
		}
		
		/**
		 * Set base directory
		 * 
		 * @param baseDir base directory
		 * @return {@link Builder}
		 */
		public Builder setBaseDir(String baseDir) {
			
			this.delegate.setBaseDir(baseDir);
			
			return this;
		}
		
		/**
		 * Set lib directory
		 * 
		 * @param libDir lib directory
		 * @return {@link Builder}
		 */
		public Builder setLibDir(String libDir) {
			
			this.delegate.setLibDir(libDir);
			
			return this;
		}
		
		/**
		 * Set data directory
		 * 
		 * @param dataDir data directory
		 * @return {@link Builder}
		 */
		public Builder setDataDir(String dataDir) {
			
			this.delegate.setDataDir(dataDir);
			
			return this;
		}
		
		/**
		 * Add arguments
		 * 
		 * @param arguments arguments
		 * @return {@link Builder}
		 */
		public Builder addArgument(@NonNull String... arguments) {
			
			for (String argument : arguments) {
				
				this.delegate.addArg(argument);
			}
			
			return this;
		}
		
		/**
		 * Build {@link MariadbServer}
		 * 
		 * @return {@link MariadbServer}
		 */
		public MariadbServer build() {
			
			return new MariadbServer(this.delegate.build());
		}
	}
}
