package com.github.shoothzj.sdk.kubectl;

import com.github.shoothzj.javatool.module.HostPort;
import com.github.shoothzj.sdk.kubectl.constant.KubectlConstant;
import com.github.shoothzj.sdk.kubectl.module.KubectlDeployModule;
import com.github.shoothzj.sdk.kubectl.module.KubectlPodModule;
import com.github.shoothzj.sdk.kubectl.module.KubectlServiceModule;
import com.github.shoothzj.sdk.kubectl.module.KubectlStatefulModule;
import com.github.shoothzj.sdk.kubectl.module.PodType;
import io.kubernetes.client.ApiClient;
import io.kubernetes.client.apis.AppsV1Api;
import io.kubernetes.client.apis.CoreV1Api;
import io.kubernetes.client.models.V1ContainerStatus;
import io.kubernetes.client.models.V1Deployment;
import io.kubernetes.client.models.V1DeploymentList;
import io.kubernetes.client.models.V1ObjectMeta;
import io.kubernetes.client.models.V1Pod;
import io.kubernetes.client.models.V1PodList;
import io.kubernetes.client.models.V1PodSpec;
import io.kubernetes.client.models.V1PodStatus;
import io.kubernetes.client.models.V1Service;
import io.kubernetes.client.models.V1ServiceList;
import io.kubernetes.client.models.V1ServicePort;
import io.kubernetes.client.models.V1StatefulSet;
import io.kubernetes.client.models.V1StatefulSetList;
import io.kubernetes.client.util.ClientBuilder;
import io.kubernetes.client.util.KubeConfig;
import lombok.extern.slf4j.Slf4j;

import java.io.FileReader;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author hezhangjian
 */
@Slf4j
public class KubeClient implements IKubectl {

    private final ApiClient client;

    private AppsV1Api appApi;

    private final CoreV1Api api;

    KubeClient() throws Exception {
        client = ClientBuilder.cluster().build();
        appApi = new AppsV1Api(client);
        api = new CoreV1Api(client);
    }

    KubeClient(String configPath) throws Exception {
        client = ClientBuilder.kubeconfig(
                KubeConfig.loadKubeConfig(new FileReader(configPath))).build();
        api = new CoreV1Api(client);
    }

    @Override
    public List<String> getStatefulHosts(String serviceName) {
        return getAllPods().stream().filter(module -> module.getContainerName().equals(serviceName))
                .map(KubectlPodModule::getHostname).collect(Collectors.toList());
    }

    @Override
    public HostPort getServiceIpPort(String serviceName) {
        KubectlServiceModule serviceModule = getServiceModule(serviceName);
        if (serviceModule == null) {
            return null;
        }
        return new HostPort(serviceModule.getClusterIp(), serviceModule.getPort());
    }

    @Override
    public int getStatefulReplicaTotal(String statefulName) {
        return getAllStatefuls().stream().filter(module -> module.getName().equals(statefulName)).map(KubectlStatefulModule::getReplicas).findAny().orElse(0);
    }

    @Override
    public int getDeployReplicaTotal(String deployName) {
        return getAllDeploys().stream().filter(module -> module.getName().equals(deployName)).map(KubectlDeployModule::getReplicas).findAny().orElse(0);
    }

    private KubectlServiceModule getServiceModule(String serviceName) {
        return getAllServices().stream().filter(module -> module.getName().equals(serviceName)).findAny().orElse(null);
    }

    private List<KubectlStatefulModule> getAllStatefuls() {
        List<KubectlStatefulModule> result = new ArrayList<>();
        try {
            V1StatefulSetList statefulSetList = appApi.listNamespacedStatefulSet(KubectlConstant.DEFAULT, true, null, null
                    , null, null, null, null, null, null);
            for (V1StatefulSet item : statefulSetList.getItems()) {
                V1ObjectMeta metadata = item.getMetadata();
                Integer integer = item.getSpec().getReplicas();
                result.add(new KubectlStatefulModule(metadata.getName(), integer));
            }
        } catch (Exception e) {
            log.error("get all stateful exception ", e);
        }
        return result;
    }

    private List<KubectlDeployModule> getAllDeploys() {
        List<KubectlDeployModule> result = new ArrayList<>();
        try {
            V1DeploymentList v1DeploymentList = appApi.listNamespacedDeployment(KubectlConstant.DEFAULT, true, null, null
                    , null, null, null, null, null, null);
            for (V1Deployment item : v1DeploymentList.getItems()) {
                V1ObjectMeta metadata = item.getMetadata();
                Integer integer = item.getSpec().getReplicas();
                result.add(new KubectlDeployModule(metadata.getName(), integer));
            }
        } catch (Exception e) {
            log.error("get all stateful exception ", e);
        }
        return result;
    }

    private List<KubectlServiceModule> getAllServices() {
        List<KubectlServiceModule> result = new ArrayList<>();
        try {
            V1ServiceList serviceList = api.listNamespacedService("default", true, null, null
                    , null, null, null, null, null, null);
            for (V1Service item : serviceList.getItems()) {
                V1ServicePort v1ServicePort = item.getSpec().getPorts().get(0);
                result.add(new KubectlServiceModule(item.getMetadata().getName(), item.getSpec().getClusterIP(),
                        v1ServicePort.getNodePort(), v1ServicePort.getPort(), v1ServicePort.getTargetPort().getIntValue()));
            }
        } catch (Exception e) {
            log.error("get all service exception ", e);
        }
        return result;
    }

    private List<KubectlPodModule> getAllPods() {
        List<KubectlPodModule> result = new ArrayList<>();
        try {
            V1PodList v1PodList = api.listNamespacedPod("default", true, null, null, null,
                    null, null, null, null, null);
            for (V1Pod item : v1PodList.getItems()) {
                V1ObjectMeta metadata = item.getMetadata();
                String kind = metadata.getOwnerReferences().get(0).getKind();
                V1PodSpec spec = item.getSpec();
                V1PodStatus status = item.getStatus();
                V1ContainerStatus containerStatus = status.getContainerStatuses().get(0);
                String containerStatusName = containerStatus.getName();
                log.info("kind is [{}] host ip is [{}], pod ip is [{}], hostname is [{}] container name is [{}]", kind, status.getHostIP(), status.getPodIP(), spec.getHostname(), containerStatusName);
                try {
                    PodType podType = PodType.valueOf(kind);
                    result.add(new KubectlPodModule(podType, status.getHostIP(), status.getPodIP(), spec.getHostname(), containerStatusName, containerStatus.isReady()));
                } catch (Exception e) {
                    log.error("may be unknown enum, protect, kind is [{}], message is [{}]", kind, e.getMessage());
                }
            }
        } catch (Exception e) {
            log.error("get all pod exception ", e);
        }
        return result;
    }

}
