package com.github.scalafanatic
package scalaalgorithms

object LongestIncreasingSubsequence {
  /**
    * Returns the largest subsequence of increasing numbers contained by a given array. Here,
    * "increasing" is defined as each element being greater than OR equal to the previous one.
    *
    * Reference: [[https://en.wikipedia.org/wiki/Longest_increasing_subsequence]]
    */
  def find(sequence: Array[Int]): Array[Int] = {
    if (sequence.isEmpty) Array.empty
    else {
      var length = 0
      val predecessorLookup = Array.ofDim[Int](sequence.length)
      // The elements of the indicesOfIncreasingSubsequenceElements array at indices {1, 2, ..., L}
      // represent pointers into the input sequence such that those elements are an increasing
      // subsequence of length L. For example, if this array is {1, 3, 5}, then that means there is
      // an increasing subsequence that can be formed using the elements at indices 1, 3, and 5 of
      // the input array.
      // For simplicity, this array does not use the 0th element so that its indices parallel the
      // length of the corresponding subsequences.
      val indicesOfIncreasingSubsequenceElements = Array.ofDim[Int](sequence.length + 1)

      def binarySearch(element: Int): Int = {
        // This binary search will find the largest element (e) in the currently-known increasing
        // subsequence such that e <= element. Because the increasing subsequence is sorted by
        // definition, this search is logarithmic in time complexity.
        var (lo, hi) = (1, length)
        while (lo <= hi) {
          val mid = Math.ceil((lo + hi).toDouble / 2d).toInt
          if (sequence(indicesOfIncreasingSubsequenceElements(mid)) <= element) {
            lo = mid + 1
          } else {
            hi = mid - 1
          }
        }
        lo
      }

      sequence.indices.foreach { i =>
        val newLength = binarySearch(sequence(i))
        predecessorLookup(i) = indicesOfIncreasingSubsequenceElements(newLength - 1)
        indicesOfIncreasingSubsequenceElements(newLength) = i
        if (newLength > length) {
          length = newLength
        }
      }

      val subsequence = Array.ofDim[Int](length)
      var k = indicesOfIncreasingSubsequenceElements(length)
      for (i <- length - 1 to 0 by -1) {
        subsequence(i) = sequence(k)
        k = predecessorLookup(k)
      }
      subsequence
    }
  }
}
